<?php

namespace Inside\Host\Normalizer\Entity;

use Drupal;
use Exception;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

/**
 * Paragraph Normalizer
 *
 * @category  Class
 * @package   Inside\Host\Normalizer\Entity\ParagraphNormalizer
 * @author    Maecia <technique@maecia.com>
 * @author    Nicolas Deniaud <contact@feldoe.net>
 * @copyright 2018 Maecia
 * @link      http://www.maecia.com/
 */
class ParagraphNormalizer extends BaseEntityNormalizer
{

    /**
     * {@inheritdoc}
     */
    protected $supportedInterfaceOrClass = \Drupal\paragraphs\ParagraphInterface::class;

    /**
     * {@inheritDoc}
     */
    protected $domain = 'Sections';

    /**
     * {@inheritdoc}
     */
    protected $removableFields = [
        'id',
        'uid',
        'revision_id',
        'type',
        'revision_uid',
        'behavior_settings',
        'default_langcode',
        'revision_default',
        'revision_translation_affected',
        'content_translation_source',
        'content_translation_outdated',
        'content_translation_changed',
        'content_translation_uid',
        'pgID',
    ];

    /**
     * {@inheritDoc}
     */
    protected $swappableNames = [
        'created' => 'created_at',
        'changed' => 'updated_at',
        'uuid' => 'uuid_host',
        'parent_id' => 'sectionable_uuid',
        'parent_type' => 'sectionable_type',
        'parent_name' => 'field',
        'uid' => 'author',
    ];

    /**
     * {@inheritDoc}
     */
    public function normalize($entity, $format = null, array $context = [])
    {
        if (!isset($context['langcode'])) {
            $context['langcode'] = $entity->language()->getId();
        }
        $attributes = parent::normalize($entity, $format, $context);

        $attributes = $this->formatDatasToNormalization($entity, $attributes);

        $attributes['bundle'] = $entity->bundle();

        $attributes['uuid'] = get_lumen_entity_uuid($entity);
        $attributes['field'] = str_replace('field_', '', $attributes['field']);

        if ($attributes['sectionable_uuid']) {
            $target = Drupal::service('entity_type.manager')->getStorage($attributes['sectionable_type'])->load(
                $attributes['sectionable_uuid']
            );

            if (!$target) {
                return [];
            }

            try {
                if ($target->hasTranslation($context['langcode'])) {
                    $target = $target->getTranslation($context['langcode']);

                    if ($target) {
                        $attributes['sectionable_type'] = get_lumen_entity_bundle($target);
                        $attributes['sectionable_uuid'] = get_lumen_entity_uuid($target);
                    }
                }
            } catch (Exception $exception) {
                Log::error($exception->getMessage());
            }
        }

        return $attributes;
    }

    /**
     * {@inheritDoc}
     */
    public function denormalize($data, $class, $format = null, array $context = [])
    {
        $values = [];
        $data = $this->prepareLangs($data, $values, 'default_langcode', 'langcode');
        $bundle = $data['bundle'];

        $values['type'] = $data['bundle'];

        unset($data['type']);
        unset($data['bundle']);
        unset($data['pgID']);

        if (isset($data['author'])) {
            unset($data['author']);
        }

        try {
            $entity = $this->createEntity($values, $data, $context, 'paragraph', 'id');

            $domain = guess_drupal_entity_type($data['sectionable_type']) === 'node' ? 'Contents' : 'Sections';

            $query = call_user_func(
                '\\Inside\\Content\\Models\\'.$domain.'\\'.Str::studly($data['sectionable_type']).'::withoutGlobalScopes'
            );
            $content = $query->findOrFail($data['sectionable_uuid']);

            $data['sectionable_type'] = guess_drupal_entity_type($data['sectionable_type']);
            $data['sectionable_uuid'] = $content->uuid_host;

            $node = Drupal::service('entity.repository')
                ->loadEntityByUuid($data['sectionable_type'], $data['sectionable_uuid']);

            if ($node) {
                $data['sectionable_uuid'] = $node->id(); // Update needs uid
            }
            $result = $this->formatDatasToDenormalization($data, 'paragraph', $bundle);

            $result['parent_field_name'] = $result['parent_name']; // TODO: check parent_field_name / parent_name
            unset($result['parent_name']);
            unset($result['uid']);

            $this->denormalizeFieldData($result, $entity, $format, $context);
        } catch (Exception $e) {
            Log::error('[ParagraphNormalizer] Paragraph denormalization failed ['.$e->getMessage().']');
            Log::debug('[ParagraphNormalizer] Paragraph denormalization failed trace ['.$e->getTraceAsString().']');
        }

        return $entity;
    }
}
