<?php

namespace Inside\Host\Normalizer\Entity;

use Drupal;
use Drupal\comment\CommentInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityTypeRepositoryInterface;
use Exception;
use Illuminate\Support\Str;
use Inside\Content\Models\Contents\Comments;
use Illuminate\Support\Facades\Log;

/**
 * Comment Normalizer
 *
 * @category  Class
 * @package   Inside\Host\Normalizer\Entity\CommentNormalizer
 * @author    Maecia <technique@maecia.com>
 * @copyright 2018 Maecia
 * @link      http://www.maecia.com/
 */
class CommentNormalizer extends BaseEntityNormalizer
{
    protected $supportedInterfaceOrClass = CommentInterface::class;

    protected $removableFields = [
        'id',
        'cid',
        //  'bundle',// THIS IS REQUIRED TO WORK
        'deleted',
        'revision_id',
        'comment_type',
        'subject',
        'mail',
        'homepage',
        'hostname',
        // 'thread', // THIS IS REQUIRED TO WORK
        'entity_type',
        'name',
        'default_langcode',
        'delta',
        'comment_body_value',
        'title',
        'comment_body_format',
        'field_name',
        'entity_id',
        'content_translation_source',
        'content_translation_outdated',
    ];

    protected $swappableNames = [
        'comment_body' => 'body',
        'created' => 'created_at',
        'changed' => 'updated_at',
        'uuid' => 'uuid_host',
        'uid' => 'author',
    ];

    public function normalize(
        $entity,
        $format = null,
        array $context = []
    ): float|array|\ArrayObject|bool|int|string|null {
        // Set context langcode needed by normalization fields
        $context['langcode'] = $entity->language()->getId();

        $data = parent::normalize($entity, $format, $context);

        $data = $this->formatDatasToNormalization($entity, $data);

        // We want to get parent entity : meaning where does this comment is shown ?
        $parent = $entity->getCommentedEntity();

        if ($parent?->hasTranslation($context['langcode'])) {
            $parent = $parent->getTranslation($context['langcode']);
        }

        $replyTo = $entity->getParentComment();
        $owner = $entity->getOwner();

        // Get our lumen uuid
        $data['uuid'] = get_lumen_entity_uuid($entity);
        if (!$data['uuid']) { // This is a creation we don't know inside entity yet
            unset($data['uuid']);
        }
        unset($data['uuid_host']);

        // We add our parent node uuid_host
        $data[$parent->getType()] = [get_lumen_entity_uuid($parent)];
        $data['bundle'] = $parent->getType();
        $data['pid'] = $replyTo ? get_lumen_entity_uuid($replyTo) : null;
        $data['author'] = $owner ? get_lumen_entity_uuid($owner) : null;

        return $data;
    }

    /**
     * {@inheritdoc}
     */
    public function denormalize($data, $class, $format = null, array $context = [])
    {
        $values = [];
        $data = $this->prepareLangs($data, $values, 'default_langcode', 'langcode');
        $parentModel = 'Inside\Content\Models\Contents\\'.Str::studly($data['bundle']);

        $entity_type = $this->getEntityTypeDefinition('comment');

        if ($entity_type->hasKey('bundle')) {
            $bundle_key = $entity_type->getKey('bundle');
            // $values[$bundle_key] = $data['bundle'] ?? 'comments';
            $values[$bundle_key] = 'comments';
            // Unset the bundle key from data, if it's there.
            unset($data[$bundle_key]);
        }

        $values['comment_type'] = 'comments';
        $values['entity_type'] = 'node';
        $values['field_name'] = 'field_comments';

        $data['subject'] = substr($data['body'], 0, 64);

        $parent = call_user_func($parentModel.'::find', $data[$data['bundle']][0]);

        if ($parent) {
            $parentEntity = Drupal::service('entity.repository')
                ->loadEntityByUuid(
                    $values['entity_type'],
                    $parent->uuid_host
                );
            if (is_null($parentEntity)) {
                Log::error(
                    '[CommentNormalizer::denormalize] failed to get parent',
                    [
                        'values' => $values,
                        'parent' => $parent,
                    ]
                );
            } else {
                $values['entity_id'] = $parentEntity->id();
            }
        }
        if (array_key_exists($data['bundle'], $data)) {
            // Remove relation
            unset($data[$data['bundle']]);
        }

        $bundle = 'comments';

        unset($data['parent']);
        unset($data['type']);
        unset($data['bundle']);
        unset($data['content_translation_source']); // Comment does not manage translation

        // Get author
        // No need should be deserialized by entity ref field
        //$author        = \Inside\Content\Models\Contents\Users::find($data['author']);
        //$authorEntity  = \Drupal::service('entity.repository')->loadEntityByUuid('user', $author->uuid_host);
        //$values['uid'] = $authorEntity->id();

        $entity = null;
        try {
            if (isset($data['uuid'])) {
                $comment = Comments::find($data['uuid']);
                if ($comment) { // Check if it's an update and get uuid_host
                    $data['uuid_host'] = $comment->uuid_host; // createEntity waits for an uuid_host key if it exists
                }
            }

            // Create a fresh comment entity or update it if we got an uuid_host
            $entity = $this->createEntity($values, $data, $context, 'comment', 'cid');
            $result = $this->formatDatasToDenormalization($data, 'comment', $bundle);

            // Result has the data correctly formated for a denormalization
            $this->denormalizeFieldData($result, $entity, $format, $context);
        } catch (Exception $e) {
            Log::error('Comment denormalization failed ['.$e->getMessage().']');
            Log::debug('Comment denormalization failed trace ['.$e->getTraceAsString().']');
        }

        return $entity;
    }
}
