<?php

namespace Inside\Host\Helpers\Adapter\Form;

use Drupal;
use Drupal\Component\Utility\Bytes;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\field\Entity\FieldConfig;
use Illuminate\Support\Facades\Log;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\DynamicClass;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Field;
use Inside\Support\HasMimeTypes;
use Inside\Support\Str;

/**
 * Flatten formDisplay from Drupal 8
 *
 * @category  Class
 * @package   Inside\Host\Adapter\Form\BaseForm
 * @author    Maecia <technique@maecia.com>
 * @copyright 2018 Maecia
 * @link      http://www.maecia.com/
 */
class BaseForm
{
    use HasMimeTypes;

    /**
     * Convert widget to match core
     *
     * @var array
     */
    protected const WIDGET_NAME = [
        'comment_default' => 'comments',
        'taxonomy_autocomplete' => 'autocomplete',
        'entity_reference_autocomplete_tags' => 'autocomplete',
        'entity_reference_autocomplete' => 'autocomplete',
        'entity_reference_revisions' => 'section',
        'entity_reference_paragraphs' => 'section',
        'options_buttons' => 'checkboxes',
        'options_select' => 'select',
        'options_onoff' => 'onoff',
        'boolean_checkbox' => 'onoff',
        'image_image' => 'image',
        'file_generic' => 'file',
        'string_textfield' => 'text',
        'string_textarea' => 'textarea',
        'text_textfield' => 'text',
        'text_textarea' => 'wysiwyg',
        'text_textarea_with_summary' => 'wysiwyg',
        'datetime_timestamp' => 'datetime',
        'inside_timestamp_field_widget' => 'inside_timestamp_field_widget',
        'inside_link_field_widget' => 'inside_link_field_widget',
        'inside_color_picker_field_widget' => 'color_picker',
        'inside_icon_picker_field_widget' => 'icon_picker',
        'inside_phone_field_widget' => 'phone',
        'list_string' => 'select',
        'list_float' => 'select',
        'list_integer' => 'select',
        'email_default' => 'email',
    ];

    /**
     * Convert field names to match core
     *
     * @var array
     */
    protected const FIELD_NAME = [];

    /**
     * Removable fields
     *
     * @var array
     */
    protected const REMOVABLE_FIELD = [];

    /**
     * Convert typeinside_section_
     *
     * @var array
     */
    protected const TYPE_NAME = [
        'string' => 'text',
        'string_long' => 'textarea',
        'text' => 'wysiwyg',
        'text_long' => 'wysiwyg',
        'text_with_summary' => 'wysiwyg',
        'entity_reference' => 'reference',
        'entity_reference_revisions' => 'section',
    ];

    /**
     * The available languages
     */
    protected array $languages = [];

    /**
     * Field group
     */
    protected array $drupalGroupsFields = [];

    /**
     * Fields
     */
    protected array $drupalFormFields = [];

    /**
     * Construct a new BaseForm instance
     */
    public function __construct(
        protected string $type,
        protected string $bundle
    ) {
        Drupal::service('inside');

        $this->languages = array_keys(Drupal::languageManager()->getLanguages());
    }

    /**
     * Create or update fields
     *
     * @param  null  $formDisplay
     * @return void
     * @throws EntityStorageException
     */
    public function setupForm($formDisplay = null): void
    {
        if ($formDisplay === null) {
            $formDisplay = Drupal::service('entity_type.manager')->getStorage('entity_form_display')->load(
                $this->type.'.'.$this->bundle.'.default'
            );
        }

        if (!$formDisplay) {
            return;
        }

        $this->drupalFormFields = $formDisplay->getComponents();
        $model = $this->getModel();

        $oldFields = Field::whereHas(
            'model',
            function ($query) use ($model) {
                $query->whereClass($model);
            }
        )->pluck('name');

        $this->drupalGroupsFields = $formDisplay->getThirdPartySettings('field_group');

        $needToDisableRebuild = DynamicClass::isRebuildAutoLoadNeeded();
        if ($needToDisableRebuild) {
            DynamicClass::disableRebuildAutoLoad();
        }

        foreach ($this->drupalFormFields as $drupalFieldName => $drupalFormFieldConfig) {
            if (in_array($drupalFieldName, $this::REMOVABLE_FIELD)) {
                continue; // Unused drupal field in Inside
            }

            // Compute inside field name
            $fieldName = str_replace('field_', '', $drupalFieldName);
            $fieldName = $this::FIELD_NAME[$fieldName] ?? $fieldName;

            $formDisplayConfig = Drupal::config('core.entity_form_display.'.$this->type.'.'.$this->bundle.'.default');
            $isHidden = array_key_exists($drupalFieldName, $formDisplayConfig->getRawData()['hidden'] ?? []);

            $fieldConfig = FieldConfig::loadByName($this->type, $this->bundle, $drupalFieldName);

            if (!$fieldConfig) {
                // This is a special drupal field ( examples: title, author, etc... )
                $this->handleAttributeFields($fieldName, $drupalFormFieldConfig, $isHidden);
                if (($key = $oldFields->search($fieldName)) !== false) {
                    $oldFields->forget($key);
                }
                continue;
            }
            $type = $fieldConfig->get('field_type');
            $type = $this::TYPE_NAME[$type] ?? $type;
            $settings = $this->getFieldSettings($type, $drupalFormFieldConfig, $fieldConfig);

            $data = [
                'model' => $model,
                'name' => $fieldName,
                'type' => $type,
                'displayed' => !$isHidden,
                'options' => $settings,
            ];

            $this->createOrUpdateField($data);
            if (($key = $oldFields->search($fieldName)) !== false) {
                $oldFields->forget($key);
            }
        }
        $this->handleCustomFields($oldFields);

        $this->handleHiddenFields($formDisplay, $oldFields);

        foreach ($oldFields as $oldField) {
            Field::whereHas(
                'model',
                function ($query) use ($model) {
                    $query->whereClass($model);
                }
            )->whereName($oldField)->delete();
        }

        if ($needToDisableRebuild) {
            DynamicClass::enableRebuildAutoLoad();
            DynamicClass::rebuild();
        }
    }

    /**
     * Create or update field
     *
     * @param  array  $data
     *
     * @return void
     */
    protected function createOrUpdateField(array $data): void
    {
        // created_at and updated_at are no more form fields => internal system fields now
        // We break sync between drupal & inside at this point but inside info are the correct ones!
        if (in_array($data['name'], ['created_at', 'updated_at'])) {
            return;
        }
        $field = Field::whereHas(
            'model',
            function ($query) use ($data) {
                $query->whereClass($data['model']);
            }
        )->where('name', $data['name'])->where('type', $data['type'])->first();

        if ($field === null) {
            try {
                $information = Schema::getModelInformation(class_to_type($data['model']));
            } catch (ModelSchemaNotFoundException) {
                // Model is not set yet !
                return;
            }

            $data['model_id'] = $information->id;
            unset($data['model']);
            Field::create($data);

            return;
        }
        unset($data['model']);

        $field->update($data);
    }

    /**
     * Add hidden fields
     *
     * @throws EntityStorageException
     */
    protected function handleHiddenFields($formDisplay, &$oldFields): void
    {
        $model = $this->getModel();
        $fields = Drupal::service('entity_field.manager')->getFieldDefinitions($this->type, $this->bundle);

        $displayedFields = Field::whereHas(
            'model',
            function ($query) use ($model) {
                $query->whereClass($model);
            }
        )->where('displayed', true)->get()->pluck('name')->toArray();

        foreach ($fields as $fieldName => $field) {
            $name = str_replace('field_', '', $fieldName);
            $name = $this::FIELD_NAME[$name] ?? $name;
            $fieldConfig = FieldConfig::loadByName($this->type, $this->bundle, $fieldName);

            if (in_array($name, $this::REMOVABLE_FIELD) || in_array($name, $displayedFields)) {
                continue;
            }

            if (!$fieldConfig) {
                $this->handleAttributeFields($name, $field, true);
                if (($key = $oldFields->search($name)) !== false) {
                    $oldFields->forget($key);
                }
                continue;
            }

            $type = $fieldConfig->get('field_type');
            $type = $this::TYPE_NAME[$type] ?? $type;
            $settings = $this->getFieldSettings($type, $field, $fieldConfig);

            $data = [
                'model' => $model,
                'name' => $name,
                'type' => $type,
                'displayed' => false,
                'options' => $settings,
            ];

            $this->createOrUpdateField($data);
            if (($key = $oldFields->search($name)) !== false) {
                $oldFields->forget($key);
            }
        }
    }

    /**
     * Get the model
     *
     * @return string
     */
    protected function getModel(): string
    {
        return 'Inside\Content\Models\Contents\\'.Str::studly($this->bundle);
    }

    /**
     * Handle attributes
     *
     * @param  string  $name
     * @param  mixed  $field
     * @param  bool  $isHidden
     */
    protected function handleAttributeFields(string $name, $field = null, bool $isHidden = false)
    {
    }

    /**
     * Return the field settings
     *
     * @param  string  $type
     * @param  mixed  $field
     * @param  FieldConfig|null  $fieldConfig
     *
     * @return array
     * @throws EntityStorageException
     */
    protected function getFieldSettings(string $type, $field = null, ?FieldConfig $fieldConfig = null): array
    {
        Drupal::service('inside');
        $fieldName = $fieldConfig->get('field_name');

        if (is_array($field)) {
            $weight = $field['weight'];
            $fieldType = $field['type'];
        } else {
            $weight = 0;
            $fieldType = $field->get('field_type');
        }

        $config = [
            'required' => (bool) ($fieldConfig->get('required') ?? true),
            'translatable' => (bool) ($fieldConfig->get('translatable') ?? true),
            'cardinality' => $fieldConfig->getFieldStorageDefinition()->getCardinality(),
            'default' => $fieldConfig->getFieldStorageDefinition()->get('cardinality') == 1 ? null : [],
            'weight' => $weight,
            'widget' => $this::WIDGET_NAME[$fieldType] ?? $fieldType,
            'group' => [],
            'searchable' => (bool) (get_third_party_setting($fieldConfig, 'inside', 'search', 'searchable')
                    ?? false)
                || (bool) (get_third_party_setting($fieldConfig, 'inside', 'search', 'global_searchable') ?? false),
            // Being global searchable and not searchable is a nonsense
            'global_searchable' => (bool) (get_third_party_setting(
                    $fieldConfig,
                    'inside',
                    'search',
                    'global_searchable'
                ) ?? false),
            'filter_widget' => get_third_party_setting($fieldConfig, 'inside', 'search', 'filter_widget') ??
                '',
            'filter_order' => (int) get_third_party_setting($fieldConfig, 'inside', 'search', 'filter_order')
                ?? 1,
            'filter_category' => get_third_party_setting($fieldConfig, 'inside', 'search', 'filter_category') ??
                '',
            'filter_query_mode' => get_third_party_setting($fieldConfig, 'inside', 'search',
                    'filter_query_mode') ?? \Search::OPERATOR_OR,
            'searchable_filter' => (bool) (get_third_party_setting(
                    $fieldConfig,
                    'inside',
                    'search',
                    'searchable_filter'
                ) ?? false),
            'search_result_field' => (bool) (get_third_party_setting(
                    $fieldConfig,
                    'inside',
                    'search',
                    'search_result_field'
                ) ?? false),
            'classifiable' => get_third_party_setting($fieldConfig, 'inside', 'classifiable') ?? '',
            'editable' => (bool) (get_third_party_setting($fieldConfig, 'inside', 'editable') ?? false),
            'categorizable' => (bool) (get_third_party_setting(
                    $fieldConfig,
                    'inside',
                    'permission',
                    'categorizable'
                ) ?? false),
            'permissible' => (bool) (get_third_party_setting(
                    $fieldConfig,
                    'inside',
                    'permission',
                    'permissible'
                ) ?? false),
            'selectable_all' => (bool) (get_third_party_setting(
                    $fieldConfig,
                    'inside',
                    'special',
                    'selectable_all'
                ) ?? false),
            'front_field_config' => get_third_party_setting(
                    $fieldConfig,
                    'inside',
                    'special',
                    'front_field_config'
                ) ?? '{}',
            'content_type_icon_class' => get_third_party_setting(
                    $fieldConfig,
                    'inside',
                    'front_parameters',
                    'content_type_icon_class'
                ) ?? '',
            'reference_is_a_parent' => get_third_party_setting(
                    $fieldConfig,
                    'inside',
                    'special',
                    'reference_is_a_parent'
                ) ?? false,
            'non_exportable' => get_third_party_setting(
                    $fieldConfig,
                    'inside',
                    'special',
                    'non_exportable'
                ) ?? false,
            'image_styles' => get_third_party_setting($fieldConfig, 'inside', 'special', 'image_styles') ??
                [],
        ];

        foreach ($this->drupalGroupsFields as $fieldGroupId => $fieldGroup) {
            if (in_array($fieldName, $fieldGroup['children'])) {
                $config['group'] = [
                    'id' => $fieldGroupId,
                    'weight' => $fieldGroup['weight'],
                    'type' => $fieldGroup['format_type'],
                    'classes' => $fieldGroup['format_settings']['classes'],
                    'collapsible' => (bool) ($fieldGroup['collapsible'] ?? true),
                    'collapsed' => (bool) ($fieldGroup['collapsed'] ?? true),
                ];
                foreach ($this->languages as $language) {
                    $config['group']['label'][$language] = (string) t(
                        trim($fieldGroup['label']),
                        [],
                        ['langcode' => $language]
                    );
                }
            }
        }

        $entityType = $fieldConfig->get('entity_type');
        $bundle = $fieldConfig->get('bundle');
        $key =
            'field.field.'.$this->type.'.'.($entityType == 'users' ? 'user' : $bundle).'.'.$fieldName;

        $configLanguage = $fieldConfig->get('langcode');
        $languageManager = Drupal::languageManager();

        // Default language
        $config['title'][$configLanguage] = $fieldConfig->get('label');
        $config['description'][$configLanguage] = $fieldConfig->get('description');
        foreach ($this->languages as $language) {
            if ($language == $configLanguage) {
                continue;
            }
            $config['title'][$language] =
                $languageManager->getLanguageConfigOverride($language, $key)->get('label');
            $config['description'][$language] =
                $languageManager->getLanguageConfigOverride($language, $key)->get('description');
        }
        if (is_array($fieldConfig->get('default_value'))) {
            foreach ($fieldConfig->get('default_value') as $default) {
                if (isset($default['value'])) {
                    $config['default'] = $default['value'];
                } else {
                    $config['default'] = $default;
                }
            }
        } else {
            $config['default'] = $fieldConfig->get('default_value');
        }

        $fieldSettings = $fieldConfig->get('settings');

        switch ($type) {
            case 'list_float':
            case 'list_integer':
            case 'list_string':
                $config['allowed_values'] = [];

                $key = 'field.storage.'.$this->type.'.'.$fieldName;
                foreach ($this->languages as $language) {
                    $allowedTranslatedValues = $languageManager->getLanguageConfigOverride($language, $key)->get(
                        'settings.allowed_values'
                    ); // Get it from storage config
                    $storageConfig = $fieldConfig->getFieldStorageDefinition()->getSettings();
                    $allowedValues = $storageConfig['allowed_values'] ?? [];

                    $i = 0;
                    foreach ($allowedValues as $allowedKey => $allowedValue) {
                        $config['allowed_values'][$language][$allowedKey] =
                            $allowedTranslatedValues[$i++]['label'] ?? $allowedValue;
                    }
                }
                break;
            case 'text':
                $config['placeholder'] = [];
                if (is_array($field)) {
                    $config['size'] = $field['settings']['size'] ?? 255;

                    if (isset($field['settings']['placeholder'])) {
                        foreach ($this->languages as $language) {
                            $config['placeholder'][$language] = (string) t(
                                trim($field['settings']['placeholder']),
                                [],
                                ['langcode' => $language]
                            );
                        }
                    }

                    switch ($field['type']) {
                        case 'inside_phone_field_widget':
                            $config['short_number_length'] = intval($field['settings']['short_number_length'] ?? '0');
                            break;
                    }
                }
                break;
            case 'image':
            case 'file':
                $extensions = explode(' ', $fieldSettings['file_extensions']);

                $mimetypes = [];
                foreach ($extensions as $extension) {
                    $mimetypes = array_merge($mimetypes, $this->getMimeTypes($extension));
                }

                $config['mimetypes'] = implode(',', array_unique($mimetypes));
                $config['extensions'] = $fieldSettings['file_extensions'];

                $maxFilesize = Bytes::toInt(Drupal\Component\Utility\Environment::getUploadMaxSize());

                if (!empty($fieldSettings['max_filesize'])) {
                    $maxFilesize = min($maxFilesize, Bytes::toInt($fieldSettings['max_filesize']));
                }

                $config['max_filesize'] = $maxFilesize;

                $host = filter_input(INPUT_SERVER, 'HTTP_HOST', FILTER_SANITIZE_URL);
                $defaultValues = $fieldConfig->get('default_value');

                if ($defaultValues) {
                    foreach ($defaultValues as $defaultValue) {
                        $file = Drupal::service('entity.repository')->loadEntityByUuid('file', $defaultValue);
                        if (!$file) {
                            break;
                        }
                        $url = Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());

                        if ($config['cardinality'] == 1) {
                            $config['default'] = substr($url, strpos($url, $host) + strlen($host));
                            break;
                        }
                        $config['default'][] = substr($url, strpos($url, $host) + strlen($host));
                    }
                }

                if ($type == 'image') {
                    $config['max_width'] = null;
                    $config['max_height'] = null;
                    $config['min_width'] = null;
                    $config['min_height'] = null;

                    $maxResolution = explode('x', $fieldSettings['max_resolution']);
                    $minResolution = explode('x', $fieldSettings['min_resolution']);

                    $default = $fieldSettings['default_image']['uuid'];

                    if ($fieldSettings['max_resolution']) {
                        $config['max_width'] = $maxResolution[0];
                        $config['max_height'] = $maxResolution[1];
                    }
                    if ($fieldSettings['min_resolution']) {
                        $config['min_width'] = $minResolution[0];
                        $config['min_height'] = $minResolution[1];
                    }
                }
                break;
            case 'comment':
                $storage = $fieldConfig->get('fieldStorage');
                $config['default'] = null;
                if (empty($storage->get('settings')['comment_type'])) {
                    Log::error('[BaseForm] Comment field storage is broken and has no comment type => fixing');
                    // Fix: probablement une migration échouée, on force le type de commentaire sur le storage
                    $storageSettings = $storage->get('settings');
                    $storageSettings['comment_type'] = 'comments';
                    $storage->set('settings', $storageSettings)->save();
                }
                $config['target'] = $storage->get('settings')['comment_type'];
                break;
            case 'section':
            case 'reference':
                $targetKey = 'target_bundles';

                if ($type == 'section') {
                    $targetKey = 'target_bundles_drag_drop';
                }
                $targetType = str_replace('default:', '', $fieldSettings['handler']);

                if (isset($fieldSettings['handler_settings'][$targetKey])) {
                    if ($targetKey === 'target_bundles_drag_drop') {
                        uasort($fieldSettings['handler_settings'][$targetKey], function ($a, $b) {
                            return $a['weight'] <=> $b['weight'];
                        });
                    }

                    $targets = [];

                    $negate = isset($fieldSettings['handler_settings']['negate'])
                    && $fieldSettings['handler_settings']['negate'] ? 1 : 0;

                    foreach ($fieldSettings['handler_settings'][$targetKey] as $key => $target) {
                        if (!isset($target['enabled'])) {
                            $enabled = true;
                        } else {
                            $enabled = $negate ? !$target['enabled'] : $target['enabled'];
                        }

                        if ($enabled) {
                            $targets[] = $key;
                        }
                    }

                    if ($targetType == 'menu_link_content') {
                        foreach ($targets as &$target) {
                            $target .= '_menus';
                        }
                    }
                    $config['target'] = $targets;
                }

                if ($targetType == 'user') {
                    $config['target'] = ['users'];
                }

                $defaultValues = $fieldConfig->get('default_value');
                if ($defaultValues) {
                    $config['default'] = [];
                    foreach ($defaultValues as $defaultValue) {
                        $config['default'][] = $defaultValue['target_uuid'];
                    }
                    unset($config['default']['target_uuid']);
                }
                break;
            case 'timestamp':
                $config['default'] = null; // Default timestamp is now, not a predefined date
                break;
        }

        return $config;
    }

    /**
     * Handle custom type fields
     *
     * @param $oldFields
     * @return void
     */
    protected function handleCustomFields(&$oldFields): void
    {
    }

    /**
     * Attach field group
     *
     * @param  string  $fieldName
     * @param  array  $options
     *
     * @return void
     */
    protected function attachGroup(string $fieldName, array &$options): void
    {
        $flip = array_flip($this::FIELD_NAME);
        $fieldName = $flip[$fieldName] ?? $fieldName;

        foreach ($this->drupalGroupsFields as $fieldGroupId => $fieldGroup) {
            if (in_array($fieldName, $fieldGroup['children'])) {
                $options['group'] = [
                    'id' => $fieldGroupId,
                    'weight' => $fieldGroup['weight'],
                    'type' => $fieldGroup['format_type'],
                    'classes' => $fieldGroup['format_settings']['classes'],
                ];
                foreach ($this->languages as $language) {
                    $options['group']['label'][$language] = (string) t(
                        trim($fieldGroup['label']),
                        [],
                        ['langcode' => $language]
                    );
                }
            }
        }
    }
}
