<?php

namespace Inside\Host\EventSubscriber;

use Drupal;
use Drupal\field\Entity\FieldConfig;
use Exception;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Content\Facades\DynamicClass;
use Inside\Content\Models\Field;
use Inside\Host\Event\Field\FieldDeleteEvent;
use Inside\Host\Event\Field\FieldInsertEvent;
use Inside\Host\Event\Field\FieldUpdateEvent;
use Inside\Host\Exceptions\ReservedFieldNameException;
use Inside\Host\Exceptions\TableNotFoundException;
use Schema;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Throwable;

/**
 * Define field events triggered in Drupal 8
 *
 * @category  Class
 * @package   Inside\Host\EventSubscriber\Field\FieldEventSubscriber
 * @author    Maecia <technique@maecia.com>
 * @copyright 2018 Maecia
 * @link      http://www.maecia.com/
 */
class FieldEventSubscriber implements EventSubscriberInterface
{
    /**
     * Prefix for each content table name
     *
     * @var string
     */
    const PREFIX = 'inside_content_';

    /**
     * Eloquent callback by fields
     *
     * @var array
     */
    const NEW_FIELD = [
        'boolean' => 'boolean',
        'timestamp' => 'dateTime',
        'datetime' => 'dateTime',
        'decimal' => 'float',
        'float' => 'float',
        'integer' => 'integer',
        'uri' => 'longText',
        'text_long' => 'longText',
        'string_long' => 'longText',
        'text_with_summary' => 'longText',
        'email' => 'string',
        'link' => 'string',
        'telephone' => 'string',
        'string' => 'string',
        'file' => 'string',
        'image' => 'string',
        'text' => 'text',
        'list_string' => 'string',
        'list_integer' => 'integer',
    ];

    /**
     * Convert table names to match core
     *
     * @var array
     */
    const TABLE_NAME = [
        'user' => 'users',
    ];

    /**
     * Convert field names to match core
     *
     * @var array
     */
    const FIELD_NAME = [
        'comment_body' => 'body',
    ];

    /**
     * Reserved names by the core
     *
     * @var array
     */
    const RESERVED_NAME = [
        'id',
        'uuid',
        'langcode',
        'pid',
        'author',
        'author_id',
        'update_author',
        'published_at',
        'children',
        'tree',
        'created_at',
        'updated_at',
    ];

    /**
     * Field callback
     *
     * @var string
     */
    protected $callback = null;

    /**
     * Field bundle
     *
     * @var string
     */
    protected $bundle = null;

    /**
     * Field table
     *
     * @var string
     */
    protected $table = null;

    /**
     * Field column
     *
     * @var string
     */
    protected $column = null;

    /**
     * Field entity type
     *
     * @var string
     */
    protected $type = null;

    /**
     * {@inheritdoc}
     */
    public static function getSubscribedEvents()
    {
        $events[FieldInsertEvent::INSERT][] = ['insert'];
        $events[FieldUpdateEvent::UPDATE][] = ['update'];
        $events[FieldDeleteEvent::DELETE][] = ['delete'];

        return $events;
    }

    /**
     * Triggered on field insert
     *
     * @param  EventSubscriberInterface  $event
     * @return void
     */
    public function insert($event): void
    {
        Drupal::service('inside');

        $field = $event->getItem();

        $this->init($field);

        if (!$this->callback || !Schema::hasTable($this->table) || Schema::hasColumn($this->table, $this->column)) {
            return;
        }

        try {
            $this->validateInput($this->table, $this->column);
        } catch (Exception $exception) {
            Log::error('[FieldEventSubscriber::insert] Failed to find table or column => '.
                $exception->getMessage());
            return;
        }

        Schema::table(
            $this->table,
            function (Blueprint $table) {
                if ($this->callback == 'string') {
                    $table->{$this->callback}($this->column, 255)->nullable();

                    return;
                }
                $table->{$this->callback}($this->column)->nullable();
            }
        );

        DynamicClass::rebuild();

        Log::info(
            'Field successfully processed.',
            ['action' => 'insert', 'column' => $this->column, 'table' => $this->table]
        );
    }

    /**
     * Triggered on field update
     *
     * @param  EventSubscriberInterface  $event
     * @return void
     */
    public function update($event): void
    {
        Drupal::service('inside');

        $field = $event->getItem();

        $this->init($field);

        if (!$this->callback) {
            return;
        }

        if (!Schema::hasColumn($this->table, $this->column)) {
            $this->insert($event);

            return;
        }

        try {
            $this->validateInput($this->table, $this->column);
        } catch (Exception $exception) {
            Log::error('[FieldEventSubscriber::update] Failed to find table or column => '.
                $exception->getMessage());
            return;
        }

        Schema::table(
            $this->table,
            function (Blueprint $table) use ($field) {
                $default = $this->getDefaultFieldValue($field);

                if ($this->callback == 'string') {
                    $table->{$this->callback}($this->column, 255)->default($default)->change();
                    $table->{$this->callback}($this->column, 255)->nullable(!$field->get('required'))->change();

                    return;
                }

                $table->{$this->callback}($this->column)->default($default)->change();
                $table->{$this->callback}($this->column)->nullable(!$field->get('required'))->change();
            }
        );

        DynamicClass::rebuild();

        Log::info(
            'Field successfully processed.',
            ['action' => 'update', 'column' => $this->column, 'table' => $this->table]
        );
    }

    /**
     * Triggered on field delete
     *
     * @param  EventSubscriberInterface  $event
     * @return void
     */
    public function delete($event): void
    {
        Drupal::service('inside');

        $field = $event->getItem();
        $domain = 'Contents';

        $this->init($field);

        if (!$this->callback || !Schema::hasColumn($this->table, $this->column)) {
            return;
        }

        try {
            $this->validateInput($this->table, $this->column);
        } catch (Exception $exception) {
            Log::error('[FieldEventSubscriber::delete] Failed to find table or column => '.
                $exception->getMessage());
            return;
        }

        Schema::table(
            $this->table,
            function (Blueprint $table) {
                $table->dropColumn($this->column);
            }
        );

        if ($this->type == 'paragraph') {
            $domain = 'Sections';
        }
        try {
            Field::whereHas(
                'model',
                function ($query) use ($domain) {
                    $query->whereClass('Inside\Content\Models\\'.$domain.'\\'.Str::studly($this->bundle));
                }
            )->where(
                [
                    'name' => $this->column,
                ]
            )->delete();
        } catch (Throwable $e) {
            Log::error('[FieldEventSubscriber] can not delete field <'.$this->column.'>');
        }

        DynamicClass::rebuild();

        Log::info(
            'Field successfully processed.',
            ['action' => 'delete', 'column' => $this->column, 'table' => $this->table]
        );
    }

    /**
     * Return default value of a given field
     *
     * @param  FieldConfig  $field
     * @return string
     */
    private function getDefaultFieldValue(FieldConfig $field)
    {
        $type = $field->get('field_type');
        $count = is_countable($field->get('default_value')) ? count($field->get('default_value')) : 0;
        $value = '';

        if ($count > 0) {
            $value = $field->get('default_value')[0]['value'];
        }

        if (in_array($type, ['decimal', 'integer'])) {
            $value = 0;
        } elseif ($type == 'float') {
            $value = 0.0;
        }

        if (!is_array($field->get('default_value'))) {
            $value = $field->get('default_value');
        }

        switch ($type) {
            case 'image':

                $uuid = data_get($field->getSettings(), 'default_image.uuid');

                if (!$uuid) {
                    return '';
                }

                $file = \Drupal::service('entity.repository')->loadEntityByUuid('file', $uuid);
                if ($file) {
                    $value = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
                    $host = filter_input(INPUT_SERVER, 'HTTP_HOST', FILTER_SANITIZE_URL);
                    return substr($value, strpos($value, $host) + strlen($host));
                }

                return '';

            case 'link':
                return isset($field->get('default_value')[0]) ? $field->get('default_value')[0]['uri'] : '';
            case 'timestamp':
                return ($value != null) ? date('Y-m-d H:m:s', $value) : null;
            default:
                return $value;
        }
    }

    /**
     * Init values
     *
     * @param  mixed  $field
     * @return void
     */
    private function init($field)
    {
        $bundle = $field->get('bundle');
        $column = str_replace('field_', '', $field->get('field_name'));

        $this->type = $field->get('entity_type');
        $this->callback = $this::NEW_FIELD[$field->get('field_type')] ?? null;
        $this->bundle = $this::TABLE_NAME[$bundle] ?? $bundle;
        $this->table = $this::PREFIX.$this->bundle;
        $this->column = $this::FIELD_NAME[$column] ?? $column;

        switch ($this->type) {
            case 'menu_link_content':
                $this->table .= '_menus';
                break;
            case 'paragraph':
                $this->table = 'inside_section_'.$this->bundle;
                break;
        }
    }

    /**
     * Perform verification before process field
     *
     * @param  string  $table
     * @param  string  $column
     * @return void
     * @throws ReservedFieldNameException
     * @throws TableNotFoundException
     */
    protected function validateInput(string $table, string $column)
    {
        if (!Schema::hasTable($table)) {
            Log::error('Table not found', ['table' => $table]);
            throw TableNotFoundException::named($table);
        }

        if (in_array($column, $this::RESERVED_NAME)) {
            Log::error('Column name is reserved.', ['column' => $column]);
            throw new ReservedFieldNameException('Column name is reserved.', 400);
        }
    }
}
