<?php
declare(strict_types=1);

namespace Inside\Host\EventSubscriber\Entity;

use Drupal;
use Drupal\Core\Entity\EntityInterface;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Event\Entity\User\UserDeleteEvent;
use Inside\Host\Event\Entity\User\UserInsertEvent;
use Inside\Host\Event\Entity\User\UserUpdateEvent;
use Inside\User\Models\User;
use Ramsey\Uuid\Uuid;

/**
 * Define user events triggered in Drupal 8
 *
 * @category  Class
 * @package   Inside\Host\EventSubscriber\User\UserEventSubscriber
 * @author    Maecia <technique@maecia.com>
 * @copyright 2018 Maecia
 * @link      http://www.maecia.com/
 */
final class UserEventSubscriber extends BaseEntityEventSubscriber
{
    protected const ENTITY_TYPE = 'user';

    public static function getSubscribedEvents(): array
    {
        $events[UserInsertEvent::INSERT][] = ['init'];
        $events[UserInsertEvent::INSERT][] = ['process'];
        $events[UserUpdateEvent::UPDATE][] = ['init'];
        $events[UserUpdateEvent::UPDATE][] = ['process'];
        $events[UserDeleteEvent::DELETE][] = ['process'];

        return $events;
    }

    /**
     * @param $event
     *
     * @throws Exception
     */
    public function init($event)
    {
        $entity     = $event->getItem();
        $serializer = Drupal::service('serializer');
        $data       = $serializer->normalize($entity);

        Log::debug('[UserEventSubscriber::init] ' . get_class($event) . ' entity (' . $entity->uuid() . ')');

        if (get_class($event) == 'Inside\Host\Event\Entity\User\UserInsertEvent') {
            $data['uuid'] = (string)Uuid::uuid4();
        } else {
            $this->uuid = null; // Reset uuid ( IMPORTANT: or during import it will use previous used uuid )
        }

        // Boot inside
        Drupal::service('inside');

        $lumenEntity = User::where('uuid_host', $entity->uuid())->first();
        if ($lumenEntity) {
            if (isset($data['uuid']) && $data['uuid'] != $lumenEntity->uuid) {
                Log::error(
                    '[UserEventSubscriber::init] Serialization conflict on (entity:' . $entity->uuid() . ',data:'
                    . $data['uuid'] . ')'
                );
            }
            $data['uuid'] = $lumenEntity->uuid;
        }

        $data['password'] = $this->getPassword($entity);

        if (get_class($event) == \Inside\Host\Event\Entity\User\UserUpdateEvent::class) {
            if (!isset($data['uuid'])) {
                Log::error('[UserEventSubscriber::init] Got UserUpdateEvent but no uuid');

                $this->uuid  = null;
                return;
            }
            if (isset($data['password'])) {
                if (empty($data['password']) || ($data['password'] === $entity->get('pass')->value)) {
                    unset($data['password']);
                }
            }

            $user = User::find($data['uuid']);

            if ($user) {
                $attributes = [
                    'name',
                    'email',
                    'password',
                    'status',
                    'langcode',
                    'api_token',
                    'remember_token',
                    'created_at',
                    'updated_at',
                ];
                $userData   = $data;

                foreach ($userData as $key => $value) {
                    if (!in_array($key, $attributes)) {
                        unset($userData[$key]);
                    }
                }

                Log::debug('[UserEventSubscriber::init] Updating user data <' . $data['uuid'] . '> with ' . json_encode($userData));
                $user->update($userData);
                $this->uuid = $user->uuid;
            } else {
                Log::error('[UserEventSubscriber::init] Can not find <' . json_encode($data) . '>');
            }

            return;
        }

        Log::debug(
            '[UserEventSubscriber::init] Creating user data <' . $data['uuid'] . '> with ' . collect($data)->only(
                [
                    'uuid',
                    'uuid_host',
                    'name',
                    'email',
                    'password',
                    'status',
                    'langcode',
                    'api_token',
                    'remember_token',
                    'created_at',
                    'updated_at',
                ]
            )->toJson()
        );
        User::create(
            collect($data)->only(
                [
                    'uuid',
                    'uuid_host',
                    'name',
                    'email',
                    'password',
                    'status',
                    'langcode',
                    'api_token',
                    'remember_token',
                    'created_at',
                    'updated_at',
                ]
            )->toArray()
        );
        $this->uuid = $data['uuid'];
    }

    /**
     * Return the password submitted (clear)
     *
     * @param \Drupal\Core\Entity\EntityInterface $user
     *
     * @return string
     */
    protected function getPassword(EntityInterface $user)
    {
        $userArray       = (array)$user;
        $clearedPassword = '';

        foreach ($userArray as $value) {
            if (is_array($value) && isset($value['pass']) && is_array($value['pass'])
                && isset($value['pass']['x-default'])
                && isset($value['pass']['x-default'][0])
                && is_array($value['pass']['x-default'][0])
                && isset($value['pass']['x-default'][0]['value'])
            ) {
                // TODO : Not really secure but Drupal doesn't permit it.
                $clearedPassword = $value['pass']['x-default'][0]['value'];
                break;
            }
        }

        return $clearedPassword;
    }

    /**
     * {@inheritDoc}
     */
    protected function cleanDataToProcess(array $data): array
    {
        // Password is only used in init
        unset($data['password']);

        return $data;
    }

    /**
     * Special case for user deletion
     */
    public function process($event): void
    {
        \Drupal::service('inside');

        $action = $event->getAction();
        $entity = $event->getItem();

        if ($action === 'delete') {
            $uuid = isset($this->uuid) ? $this->uuid : get_lumen_entity_uuid($entity);

            if ($uuid) {
                // Before deleting this user, we MUST change every content where this user is the author
                $this->changeAuthorOnContent($uuid);

                // Delete auth user
                $lumenUser = User::find($uuid);

                if ($lumenUser) {
                    $lumenUser->delete();
                }
            } else {
                Log::warning('[UserEventSubscriber:process] delete => no uuid found');
                return;
            }
        }

        // Finally proccess and delete with ContentManager
        parent::process($event);
    }

    /**
     * Change author $uuid to first maecia one
     *
     * @param $uuid
     */
    protected function changeAuthorOnContent($uuid)
    {
        $newAuthor = DB::table('inside_users')->where('email', 'like', '%maecia.com')->pluck('uuid')->first();

        if (!$newAuthor) {
            return;
        }

        $models = Schema::models();
        foreach ($models as $model => $information) {
            $query = call_user_func($information['class'] . '::query');
            $query->where('author', $uuid)->update(['author' => $newAuthor]);
        }
    }

    protected function revertAfterFailing(array $data): void
    {
        $uuidHost = $data['uuid_host'];
        if ($uuidHost && !Users::where('uuid_host', $uuidHost)->exists()) {
            Log::debug('Deleting inside user uuid host [' . $uuidHost . '] because it failed to be created in drupal');
            // Delete inside_users info, it should not exists without it's inside content users mirror
            User::where('uuid_host', $uuidHost)->delete();
        }
    }
}
