<?php
declare(strict_types=1);

namespace Inside\Host\EventSubscriber\Entity;

use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Content\Events\CommentCreatedEvent;
use Inside\Content\Events\CommentDeletedEvent;
use Inside\Content\Models\Contents\Comments;
use Inside\Host\Event\Entity\Comment\CommentDeleteEvent;
use Inside\Host\Event\Entity\Comment\CommentInsertEvent;
use Inside\Host\Event\Entity\Comment\CommentUpdateEvent;

/**
 * Define comment events triggered in Drupal 8
 *
 * @category  Class
 * @package   Inside\Host\EventSubscriber\Comment\CommentEventSubscriber
 * @author    Maecia <technique@maecia.com>
 * @copyright 2018 Maecia
 * @link      http://www.maecia.com/
 */
final class CommentEventSubscriber extends BaseEntityEventSubscriber
{
    protected const ENTITY_TYPE = 'comment';

    public static function getSubscribedEvents(): array
    {
        $events[CommentInsertEvent::INSERT][] = ['process'];
        $events[CommentInsertEvent::INSERT][] = ['addPivot'];
        $events[CommentInsertEvent::INSERT][] = ['processed'];
        $events[CommentUpdateEvent::UPDATE][] = ['process'];
        $events[CommentDeleteEvent::DELETE][] = ['removeRelations'];

        return $events;
    }

    /**
     * Comment has been created and we want to attach the comment to the content
     * via its pivot
     *
     * @param $event
     */
    public function addPivot($event): void
    {
        \Drupal::service('inside');

        $entity     = $event->getItem();
        $serializer = \Drupal::service('serializer');

        // Get normalized data from our entity
        $datas = $serializer->normalize($entity);

        // Get our attached bundle type
        $nodeBundle = $datas['bundle'];
        $nodes       = $datas[$nodeBundle];


        // Load comment
        try {
            // Load our comment
            if (isset($datas['uuid'])) {
                $comment = Comments::findOrFail($datas['uuid']);
            } else {
                $comment = Comments::where('uuid_host', (string)$entity->uuid())->firstOrFail();
            }
        } catch (ModelNotFoundException $e) {
            Log::error(
                "[CommentEventSubscriber::addPivot] failed to load Comment [" . ($datas['uuid'] ?? 'uuid not set') . "]"
            );

            return;
        }

        // Attach the comment to each node
        foreach ($nodes as $uuid) {
            try {
                if ($nodeBundle === 'user') {
                    $nodeBundle = 'users';
                }

                $query = call_user_func(
                    'Inside\\Content\\Models\\Contents\\' . Str::studly($nodeBundle) . '::query'
                );

                // Load content where the comment is to be attached
                $content = $query->findOrFail($uuid);
            } catch (Exception $e) {
                Log::error("[CommentEventSubscriber::addPivot] failed to load Content [$nodeBundle] [$uuid]");

                return;
            }

            // Attach the comment to this content
            $content->comments()->attach(
                $comment,
                [
                    'parent_type'      => get_class($content),
                    'parent_langcode'  => $content->langcode,
                    'related_type'     => Comments::class,
                    'related_langcode' => $comment->langcode,
                    'related_field'    => 'comments',
                ]
            );

            // Fire special event to let inside knows that a comment has been created and is now ready to be used

            event(new CommentCreatedEvent($comment, $content));
        }
    }

    public function removeRelations($event): void
    {
        \Drupal::service('inside');

        $entity     = $event->getItem();
        $serializer = \Drupal::service('serializer');

        // Get normalized data from our entity
        $datas = $serializer->normalize($entity);

        // Get our attached bundle type
        $nodeBundle = $datas['bundle'];
        $nodes       = $datas[$nodeBundle];

        // Load comment
        try {
            // Load our comment
            if (isset($datas['uuid'])) {
                $comment = Comments::findOrFail($datas['uuid']);
            } else {
                $comment = Comments::where('uuid', (string)$entity->uuid())->firstOrFail();
            }
        } catch (ModelNotFoundException $e) {
            Log::error(
                "[CommentEventSubscriber::removeRelations] failed to load Comment [" . ($datas['uuid'] ?? 'uuid_source not set') . "]"
            );

            return;
        }

        // Detach the comment from each node
        foreach ($nodes as $uuid) {
            try {
                if ($nodeBundle === 'user') {
                    $nodeBundle = 'users';
                }

                $query = call_user_func(
                    'Inside\\Content\\Models\\Contents\\' . Str::studly($nodeBundle) . '::query'
                );

                // Load content where the comment is to be detached
                $content = $query->findOrFail($uuid);
            } catch (Exception $e) {
                Log::error("[CommentEventSubscriber::removeRelations] failed to load Content [$nodeBundle] [$uuid]");

                return;
            }

            // Detach the comment from this content
            $content->comments()->detach(
                $comment,
                [
                    'parent_type'      => get_class($content),
                    'parent_langcode'  => $content->langcode,
                    'related_type'     => Comments::class,
                    'related_langcode' => $comment->langcode,
                    'related_field'    => 'comments',
                ]
            );

            event(new CommentDeletedEvent($comment, $content));
        }
    }
}
