<?php
declare(strict_types=1);

namespace Inside\Host\EventSubscriber\Entity;

use Drupal;
use Drupal\Core\Database\TransactionException;
use Drupal\node\Entity\Node;
use Drupal\paragraphs\Entity\Paragraph;
use Exception;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Inside\Content\Events\ContentProcessedEvent;
use Inside\Content\Facades\Schema;
use Inside\Content\Services\Managers\ContentManager;
use Inside\Content\Services\Managers\SectionManager;
use Inside\Host\Event\BaseEvent;
use Log;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Base event subscriber
 *
 * @category  Class
 * @package   Inside\Host\EventSubscriber\BaseEventSubscriber
 * @author    Maecia <technique@maecia.com>
 * @copyright 2018 Maecia
 * @link      http://www.maecia.com/
 */
abstract class BaseEntityEventSubscriber implements EventSubscriberInterface
{

    /**
     * The entity type
     *
     * @var string
     */
    protected const ENTITY_TYPE = '';

    /**
     * UUID used for user purpose only
     */
    protected ?string $uuid = null;


    /**
     * Process entity
     * @throws TransactionException
     */
    public function process(BaseEvent $event): void
    {
        Drupal::service('inside');

        $action = $event->getAction();
        $entity = $event->getItem();
        $bundle = get_lumen_entity_bundle($entity);
        $data = [];

        try {
            $serializer = Drupal::service('serializer');
            $data = $serializer->normalize($entity);

            \Illuminate\Support\Facades\Log::debug(__(
                '[BaseEntityEventSubscriber::process] processing entity :class',
                [
                    'class' => get_class($this),
                ]
            ),
                [
                    'data' => $data
                ]);

            if (!isset($data['type'])) {
                $data['type'] = static::ENTITY_TYPE;
            }

            // Little trick to get uuid_host on creation
            // Note: Normalize drop uuid_host to uuid but we are creating inside entity so uuid is not known yet
            $data['uuid_host'] = $entity->uuid();

            if ($this->uuid) {
                $data['uuid'] = $this->uuid;
            } elseif ($action == 'update') {
                $data['uuid'] = get_lumen_entity_uuid($entity);
            }

            $isNew = false;

            if (!array_key_exists('uuid', $data) || empty($data['uuid'])) {
                $isNew = true;
            }

            // Check for anonymous user
            if (array_key_exists('author', $data) && $data['author'] === null) {
                // We assume this is drupal anonymous

                // Let's put our first super admin user ( should be technique@maecia.com )
                $data['author'] = DB::table('inside_users')
                    ->where('email', 'like', '%maecia.com')
                    ->pluck('uuid')
                    ->first();
            }

            // Clean data
            $data = $this->cleanDataToProcess($data);

            $manager = new ContentManager($bundle, $data);

            if (static::ENTITY_TYPE != 'paragraph' && !$this->isEmptyContent($data)) {
                $this->forceParagraphSaveOrDelete($data, $entity);
            }

            if (static::ENTITY_TYPE == 'paragraph') {
                if (!array_key_exists('sectionable_uuid', $data) || empty($data['sectionable_uuid'])) {
                    return;
                }

                // TODO: check not always content!
                $requestData = request()->get('content');

                if (is_array($requestData)) {
                    foreach ($requestData as $requestParagraph) {
                        $requestParagraphUuid = $requestParagraph['uuid'] ?? $requestParagraph['pgID'];
                        if (0 === preg_match('/[\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12}/',
                                $requestParagraphUuid)) {
                            continue;
                        }
                        // Manage sub paragraphs
                        if (array_key_exists('content', $requestParagraph) && $data['uuid'] === $requestParagraphUuid) {
                            // Compute subparagraphs
                            $data['content'] = [];
                            foreach ($requestParagraph['content'] as $subParagraph) {
                                $subParagraphUuid = $subParagraph['uuid'] ?? $subParagraph['pgID'];
                                if (1 === preg_match('/^[\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12}$/',
                                        $subParagraphUuid)) {
                                    $data['content'][] = $subParagraphUuid;
                                }
                            }

                            $this->forceParagraphSaveOrDelete($data, $entity);
                        }
                    }
                }

                $data['sectionable_type'] = get_lumen_entity_model_name($entity->getParentEntity());
                if (0 === preg_match('/[\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12}/',
                        $data['sectionable_uuid'])) {
                    $this->guessSectionableContent($data);
                }

                $this->setParagraphWeight($data, $entity);

                $manager = new SectionManager($bundle, $data);
            }

            $manager->{$action}();

            if (static::ENTITY_TYPE != 'paragraph' && !$this->isEmptyContent($data)) {
                $this->forceSyncParagraphs($data, $entity);
            }

            if ($isNew && $entity->getEntityTypeId() === 'node') {
                $uuid = get_lumen_entity_uuid($entity);

                $query = call_user_func(type_to_class($bundle).'::withoutGlobalScopes');
                $model = $query->find($uuid);

                if ($model) {
                    event(new \Inside\Content\Events\ContentInsertedEvent($model));
                }
            }
        } catch (Exception $e) {
            if (php_sapi_name() !== 'cli' && isset($GLOBALS['bootDrupal']) && $GLOBALS['bootDrupal']) {
                Drupal::messenger()->addMessage($e->getMessage(), 'error');
                $response = new RedirectResponse(Drupal::request()->getRequestUri());
                $response->send();
            }

            Log::error('Error while processing entity to database.',
                ['action' => $action, 'type' => $bundle, 'data' => $data, 'message' => $e->getMessage()]);

            $this->revertAfterFailing($data);
            throw new TransactionException($e->getMessage(), 400);
        }

        if (isset($data['sectionable_uuid']) && 0 === preg_match('/[\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12}/',
                $data['sectionable_uuid'])) {
            Log::warning(__(
                ':trace (:manager)',
                [
                    'manager' => get_class($manager),
                    'trace' => collect(debug_backtrace())->pluck('line', 'file'),
                ]
            ));
        }
        Log::info('Entity successfully processed.', [
            'action' => $action,
            'type' => $bundle,
            'data' => $data,
            'user' => App::runningInConsole() ? 'cli' : Auth::user()->uuid
        ]);
    }


    public function processed($event): void
    {
        Drupal::service('inside');

        $entity = $event->getItem();
        $serializer = Drupal::service('serializer');

        $bundle = get_lumen_entity_bundle($entity);
        if (!$bundle) {
            return;
        }
        // Get normalized data from our entity
        $datas = $serializer->normalize($entity);
        try {
            $query = call_user_func('Inside\Content\Models\\Contents\\'.Str::studly($bundle).'::withoutGlobalScopes');

            if (isset($datas['uuid'])) {
                $model = $query->findOrFail($datas['uuid']);

                ContentProcessedEvent::dispatch($model);
            }
        } catch (Exception $e) {
            Log::error($e->getMessage());
        }
    }

    public function inserted($event): void
    {
        Drupal::service('inside');

        $entity = $event->getItem();
        $serializer = Drupal::service('serializer');

        $bundle = get_lumen_entity_bundle($entity);
        // Get normalized data from our entity
        $datas = $serializer->normalize($entity);
    }

    /**
     * Because paragraph are saved before entity, their weight or deletion aren't set correctly
     *
     * @param  array  $data
     * @param  mixed  $entity
     *
     * @return void
     */
    protected function forceParagraphSaveOrDelete($data, $entity): void
    {
        $fields = Drupal::service('entity_field.manager')
            ->getFieldDefinitions(static::ENTITY_TYPE, get_entity_bundle($entity));
        $sections = Schema::getSectionTypes();

        foreach ($fields as $field) {
            if ($field->getType() == 'entity_reference_revisions') {
                $values = $entity->get($field->getName())->getValue();
                $lumenFieldName = str_replace('field_', '', $field->getName());
                if (!array_key_exists($lumenFieldName, $data)) {
                    continue; // No change on sections
                }
                $uuids = $data[$lumenFieldName];

                foreach ($values as $index => $value) {
                    $paragraph = Paragraph::load($value['target_id']);

                    if (empty($paragraph)) {
                        continue;
                    }

                    $model = get_lumen_entity_model_name($paragraph);

                    if (!isset($data['langcode']) && !is_null($paragraph->language())) {
                        $data['langcode'] = $paragraph->language()->getId();
                    }
                    $query = call_user_func($model.'::withoutGlobalScopes');
                    $content = $query->where('uuid_host', $paragraph->uuid())
                        ->where('langcode', $data['langcode'])
                        ->first();

                    if ($content) {
                        $content->weight = $index;
                        $content->save();
                    }

                    // Convert $uuids of inside uuids in drupal uuids
                    $uuids = collect($uuids)->map(function ($uuid) use ($model) {
                        $existing = call_user_func($model.'::withoutGlobalScopes')->where('uuid', $uuid)->first();

                        return $existing ? $existing->uuid_host : $uuid;
                    })->filter()->toArray();
                }

                foreach ($sections as $section) {
                    $query = call_user_func(section_type_to_class($section).'::withoutGlobalScopes');
                    $existings = $query->where([
                        'sectionable_uuid' => $data['uuid'],
                        'field' => $lumenFieldName,
                        'langcode' => $data['langcode'],
                    ])->get();

                    // delete all missing paragraph
                    foreach ($existings as $existing) {
                        if (!in_array($existing->uuid_host, $uuids)) {
                            \Illuminate\Support\Facades\Log::debug(__(
                                '[BaseEntityEventSubscriber::forceParagraphSaveOrDelete] deleting removed section :type<:id>',
                                [
                                    'type' => class_to_type($existing),
                                    'id' => $existing->uuid ?? 'null',
                                ]
                            ));
                            $existing->delete();
                        }
                    }
                }
            }
        }
    }

    /**
     * Delete translations when deleting original translation
     *
     * @param  EventSubscriberInterface  $event
     *
     * @return void
     */
    public function deleteTranslations($event): void
    {
        Drupal::service('inside');

        $entity = $event->getItem();
        $bundle = $entity->getType();

        $languages = array_keys($entity->getTranslationLanguages());

        // Remove current langcode, it's already in the delete process
        unset($languages[$entity->langcode->value]);
        foreach ($languages as $langcode) {
            $translation = $entity->getTranslation($langcode);

            if ($translation) {
                $serializer = Drupal::service('serializer');
                $datas = $serializer->normalize($translation);

                $manager = new ContentManager($bundle, $datas);
                $manager->delete();
            }
        }
    }


    /**
     * Clean data before sending to ContentManager
     *
     * Default behavior is to send this data unchanged
     * You can overload this function to clean your data
     */
    protected function cleanDataToProcess(array $data): array
    {
        return $data;
    }

    protected function isEmptyContent($data): bool
    {
        // prevent delete paragraph when not sent in the payload - sop/rop update
        return !empty($data['content']) && empty($data['content'][0]);
    }

    protected function forceSyncParagraphs($data, $entity): void
    {
        $fields = Drupal::service('entity_field.manager')
            ->getFieldDefinitions(static::ENTITY_TYPE, get_entity_bundle($entity));

        foreach ($fields as $field) {
            if ($field->getType() == 'entity_reference_revisions') {
                $values = $entity->get($field->getName())->getValue();
                $uuids = $data[str_replace('field_', '', $field->getName())];

                foreach ($values as $index => $value) {
                    $paragraph = Paragraph::load($value['target_id']);

                    if (empty($paragraph)) {
                        continue;
                    }

                    $model = get_lumen_entity_model_name($paragraph);

                    $query = call_user_func($model.'::withoutGlobalScopes');
                    $content = $query->where('uuid_host', $paragraph->uuid())
                        ->where('langcode', $entity->language()->getId())
                        ->first();

                    if (!is_null($content) && isset($content->sectionable_uuid)) {
                        $query = call_user_func($content->sectionable_type.'::withoutGlobalScopes');
                        $parent = $query->find($content->sectionable_uuid);
                        if (!$parent) {
                            // We didn't find sectionable let's see if it is store with a nid
                            $type = guess_drupal_entity_type($content->sectionable_type);
                            $target = Drupal::service('entity_type.manager')
                                ->getStorage($type)
                                ->load($content->sectionable_uuid);
                            if ($target) {
                                // We got the target using nid let's transform to uuid
                                $targetQuery = call_user_func($content->sectionable_type.'::withoutGlobalScopes');
                                $parent = $targetQuery->where('uuid_host', $target->uuid())->first();
                                if ($parent) {
                                    // We found the correct one let's change drupal nid to inside uuid
                                    $content->sectionable_uuid = $parent->uuid;
                                    $content->save();
                                }
                            }
                        }
                    } else {
                        Log::debug(__(
                            '[BaseEntityEventSubscriber::forceSyncParagraphs] failed to get section :type host<:id>  for content :contentType <:contentId></:contentId>',
                            [
                                'type' => class_to_type($model),
                                'id' => $paragraph->uuid(),
                                'contentType' => $content->sectionable_type ?? 'unknown',
                                'contentId' => $content->sectionable_id ?? 'null',
                            ]
                        ));
                    }
                }
            }
        }
    }

    /**
     * revert things after failing to create entity
     */
    protected function revertAfterFailing(array $data): void
    {
    }

    protected function guessSectionableContent(array &$data): void
    {
        $sectionableType = class_to_type($data['sectionable_type']);
        if (empty($sectionableType)) {
            Log::error(__(
                '[BaseEntityEventSubscriber::guessSectionableContent] failed to get class type for :type',
                [
                    'type' => $data['sectionable_type'] ?? 'unset',
                ]
            ));
            return;
        }
        if (Schema::isSectionType($sectionableType)) {
            $paragraph = Paragraph::load($data['sectionable_uuid']);
            if (!is_null($paragraph)) {
                $query = call_user_func(section_type_to_class($sectionableType).'::withoutGlobalScopes');
                if (!empty($data['langcode'])) {
                    $query->where('langcode', $data['langcode']);
                }
                $query->where('uuid_host', $paragraph->uuid());
                if ($query->exists()) {
                    $data['sectionable_uuid'] = $query->first()?->uuid;
                }
            }
        } elseif (Schema::isContentType($sectionableType)) {
            $node = Node::load($data['sectionable_uuid']);
            if (!is_null($node)) {
                $query = call_user_func(type_to_class($sectionableType).'::withoutGlobalScopes');
                if (!empty($data['langcode'])) {
                    $query->where('langcode', $data['langcode']);
                }
                $query->where('uuid_host', $node->uuid());
                if ($query->exists()) {
                    $data['sectionable_uuid'] = $query->first()?->uuid;
                }
            }
        }
    }
}
