<?php

namespace Inside\Host\Bridge\Traits;

use Drupal;
use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\Entity\NodeType;
use Drupal\paragraphs\Entity\ParagraphsType;
use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Contents\ImageStyles;
use Inside\Host\Exceptions\ColumnAlreadyExistsException;

/**
 * Manage Field
 */
trait ManageField
{

    /**
     * Allowed field option and its type
     */
    protected array $fieldOptions = [
        'cardinality' => [
            'setting' => false,
            'type' => 'integer',
        ],
        'translatable' => [
            'setting' => false,
            'type' => 'boolean',
        ],
        'required' => [
            'setting' => false,
            'type' => 'boolean',
        ],
        'weight' => [
            'setting' => false,
            'type' => 'integer',
        ],
        'default' => [
            'setting' => false,
            'type' => 'mixed',
        ],
        'searchable' => [
            'setting' => true,
            'type' => 'boolean',
            'group' => 'search',
        ],
        'global_searchable' => [
            'setting' => true,
            'type' => 'boolean',
            'group' => 'search',
        ],
        'filter_widget' => [
            'setting' => true,
            'type' => 'string',
            'group' => 'search',
        ],
        'filter_order' => [
            'setting' => true,
            'type' => 'integer',
            'group' => 'search',
        ],
        'filter_category' => [
            'setting' => true,
            'type' => 'string',
            'group' => 'search',
        ],
        'filter_query_mode' => [
            'setting' => true,
            'type' => 'integer',
            'group' => 'search',
        ],
        'searchable_filter' => [
            'setting' => true,
            'type' => 'boolean',
            'group' => 'search',
        ],
        'search_result_field' => [
            'setting' => true,
            'type' => 'boolean',
            'group' => 'search',
        ],
        'classifiable' => [
            'setting' => true,
            'type' => 'string',
            'group' => null,
        ],
        'editable' => [
            'setting' => true,
            'type' => 'boolean',
            'group' => null,
        ],
        'categorizable' => [
            'setting' => true,
            'type' => 'boolean',
            'group' => 'permission',
        ],
        'permissible' => [
            'setting' => true,
            'type' => 'boolean',
            'group' => 'permission',
        ],
        'selectable_all' => [
            'setting' => true,
            'type' => 'boolean',
            'group' => 'special',
        ],
        'front_field_config' => [
            'setting' => true,
            'type' => 'string',
            'group' => 'special',
        ],
        'reference_is_a_parent' => [
            'setting' => true,
            'type' => 'boolean',
            'group' => 'special',
        ],
        'non_exportable' => [
            'setting' => true,
            'type' => 'boolean',
            'group' => 'special',
        ],
        'image_styles' => [
            'setting' => true,
            'type' => 'array',
            'group' => 'special',
        ],
        'content_type_icon_class' => [
            'setting' => true,
            'type' => 'string',
            'group' => 'front_parameters',
        ],
    ];


    /**
     * Create or update a field!
     *
     * @param          $type
     * @param          $fieldName
     * @param          $options
     * @param  int|null  $weight
     * @param  string  $entityTypeId
     * @return bool
     * @throws ColumnAlreadyExistsException
     * @throws EntityStorageException
     */
    public function contentTypeCreateOrUpdateField(
        $type,
        $fieldName,
        $options,
        int $weight = null,
        string $entityTypeId = 'node'
    ): bool {
        // fallback on users
        if ($type === 'users') {
            $entityTypeId = 'user';
        }

        if (FieldConfig::loadByName(
            $entityTypeId,
            $type == 'users' ? 'user' : $type,
            $fieldName == 'body' ? $fieldName : 'field_'.$fieldName
            ) != null) {
            return $this->contentTypeUpdateField($type, $fieldName, $options, $weight, $entityTypeId);
        }

        return $this->contentTypeCreateField($type, $fieldName, $options, $weight, $entityTypeId);
    }

    /**
     * Create a new field!
     *
     * @param          $type
     * @param          $fieldName
     * @param          $options
     * @param  null|int  $weight
     * @param  string  $entityTypeId
     * @return bool
     * @throws ColumnAlreadyExistsException
     * @throws EntityStorageException
     * @throws Exception
     */
    public function contentTypeCreateField($type, $fieldName, $options, $weight = null, $entityTypeId = 'node'): bool
    {
        // fallback on users
        if ($type === 'users') {
            $entityTypeId = 'user';
        }
        $this->write(
            '<info>Création du champ <fg=blue>'.$fieldName.'</fg=blue> au type <fg=yellow>'.$type
            .'</fg=yellow></info>'
        );

        switch ($entityTypeId) {
            case 'node':
                if (!NodeType::load($type)) {
                    throw new Exception('Node type ['.$type.'] does not exist');
                }
                break;
            case 'paragraph':
                if (!ParagraphsType::load($type)) {
                    throw new Exception('Paragraph type ['.$type.'] does not exist');
                }
                break;
        }

        try {
            $fieldSettings = $this->prepareFieldSettings($options);
        } catch (Exception $exception) {
            $this->writeln(' <fg=red>✘</fg=red>');
            Log::error($exception->getMessage());

            return false; // Fail to prepare field settings
        }

        $overrideLangCode = null;

        // Try to get field storage
        $drupalFieldName = $fieldName;
        if (!in_array($drupalFieldName, ['status', 'body', 'author', 'update_author', 'published_at'])) {
            $drupalFieldName = 'field_'.$fieldName;
        }

        $fieldStorage = FieldStorageConfig::loadByName($entityTypeId, $drupalFieldName);
        if ($fieldStorage === null) {
            $data = array_merge(
                [
                    'field_name' => $drupalFieldName,
                    'entity_type' => $entityTypeId,
                    'type' => $this->getDrupalTypeFromInside($options['type']),
                    'settings' => [],
                    'langcode' => $this->defaultLanguage,
                ],
                $this->prepareFieldStorageSettings($options)
            );
            $fieldStorage = FieldStorageConfig::create($data);
            if (array_key_exists('allowed_values', $data)) {
                // Set global allowed values ( default language )
                $fieldStorage->setSetting('allowed_values', Arr::pluck($data['allowed_values'], 'label', 'value'));
            }
            $fieldStorage->save();
        } else {
            if ($this->getDrupalTypeFromInside($options['type']) != $fieldStorage->getType()) {
                throw new Exception(
                    'Storage type ['.$type.'] ('.$fieldName.') can not be changed Type ['.$options['type']
                    .'] => ('.$this->getDrupalTypeFromInside($options['type']).') drupal ['
                    .$fieldStorage->getType().']'
                );
            }
            // Settings can't be changed either but we don't care ...

            // Check for language
            if ($this->defaultLanguage != $fieldStorage->get('langcode')) {

                // Try to delete or this will messed up
                if ($fieldStorage->isDeletable()) {
                    $fieldStorage->delete();
                    $data = [
                        'field_name' => $drupalFieldName,
                        'entity_type' => $entityTypeId,
                        'type' => $this->getDrupalTypeFromInside($options['type']),
                        'settings' => [],
                        'langcode' => $this->defaultLanguage,
                    ];
                    $fieldStorage = FieldStorageConfig::create(
                        array_merge($data, $this->prepareFieldStorageSettings($options))
                    );
                    $fieldStorage->save();
                }

                $overrideLangCode = $fieldStorage->get('langcode');
            }
        }

        if (FieldConfig::loadByName($entityTypeId, $type == 'users' ? 'user' : $type, $drupalFieldName) != null) {
            throw ColumnAlreadyExistsException::named($type, $fieldName);
        }

        // Check if node type $type is translatable
        $translatable = false;
        try {
            $contentTranslationManager = Drupal::service('content_translation.manager');
            $translatable = $contentTranslationManager->isEnabled($entityTypeId, $type);
        } catch (Exception $exception) {
            Log::error('[ManageField::contentTypeCreateField] failed to check translatable => '.$exception->getMessage());
        }

        $configLanguage = $overrideLangCode ?? $this->defaultLanguage;

        // Manage storage options
        $config = [
            'field_storage' => $fieldStorage,
            'field_name' => $drupalFieldName,
            'entity_type' => $entityTypeId,
            'bundle' => $type == 'users' ? 'user' : $type,
            'label' => $options['title'][$configLanguage] ?? $fieldName,
            'description' => $options['description'][$configLanguage] ?? '',
            'required' => $options['required'] ?? false,
            // We only allow a field to be translatable if the content type is translatable as well
            'translatable' => $translatable && $options['translatable'],
            'settings' => $fieldSettings,
            'langcode' => $overrideLangCode ?? $this->defaultLanguage,
        ];

        if (isset($options['default'])) {
            $config['default_value'] = $options['default'];
        }
        $fieldConfig = FieldConfig::create($config);

        $fieldConfig->save();

        // Third party settings
        foreach ($this->getFieldSettingsFromOptions($options) as $settingKey => $settingValue) {
            $fieldConfig->setThirdPartySetting('inside', $settingKey, $settingValue);
        }
        $fieldConfig->save();

        // Manage translations
        $languageManager = Drupal::languageManager();

        // build lang key
        $key = 'field.field.'.$entityTypeId.'.'.($type == 'users' ? 'user' : $type).'.'.$drupalFieldName;

        // Save label for each language
        foreach ($this->languages as $language) {
            if ($language == $configLanguage) {
                continue;
            }
            if (array_key_exists('title', $options) && array_key_exists($language, $options['title'])) {
                $languageManager->getLanguageConfigOverride($language, $key)->set(
                    'label',
                    $options['title'][$language]
                )->save();
            }
            if (array_key_exists('description', $options) && array_key_exists($language, $options['description'])) {
                $languageManager->getLanguageConfigOverride($language, $key)->set(
                    'description',
                    $options['description'][$language]
                )->save();
            }
        }

        // Save label for each language
        $key = 'field.storage.'.$entityTypeId.'.'.$drupalFieldName;
        foreach ($this->languages as $language) {
            $settings = $this->prepareFieldStorageSettings($options, $language);
            if (!empty($settings) && isset($settings['allowed_values'])) {
                $languageManager->getLanguageConfigOverride($language, $key)->set(
                    'settings.allowed_values',
                    array_values($settings['allowed_values'])
                )->save();
            }
        }
        $fieldConfig->save();

        // Manage form display !
        if ($options['widget'] && is_array($options['widget']) && array_key_exists('type', $options['widget'])) {
            $formDisplay = \Drupal::service('entity_display.repository')->getFormDisplay(
                $entityTypeId,
                $type == 'users' ? 'user' : $type
            );
            if (array_key_exists('start', $options['widget'])) {
                $weight += (int) $options['widget']['start'];
            }
            if ($weight === null && isset($options['widget']['weight'])) {
                $weight = $options['widget']['weight'];
            }
            $formDisplay->setComponent(
                $drupalFieldName,
                [
                    'type' => $this->getDrupalWidgetFromInside($options['widget']['type']),
                    'settings' => $options['widget']['settings'] ?? [],
                    'weight' => $weight,
                ]
            )->save();

            $this->attachFieldToGroup($formDisplay, $options, $type, $fieldName);

            // Hide ?
            if (isset($options['widget']['hidden']) && $options['widget']['hidden']) {
                $formDisplay->removeComponent($drupalFieldName);
                $formDisplay->save();
            }
            $fieldConfig->save();
        }
        $this->writeln(' <fg=green>✔</fg=green>');

        return true;
    }

    /**
     * Update an existing field
     *
     * @param          $type
     * @param          $fieldName
     * @param          $options
     * @param  null|int  $weight
     * @param  string  $entityTypeId
     * @return bool
     * @throws EntityStorageException
     * @throws Exception
     */
    public function contentTypeUpdateField($type, $fieldName, $options, $weight = null, $entityTypeId = 'node'): bool
    {
        // fallback on users
        if ($type === 'users') {
            $entityTypeId = 'user';
        }

        $this->write(
            '<info>Modification du champ <fg=blue>'.$fieldName.'</fg=blue> au type <fg=yellow>'.$type
            .'</fg=yellow></info>'
        );

        switch ($entityTypeId) {
            case 'node':
                if (!NodeType::load($type)) {
                    throw new Exception('Node type ['.$type.'] does not exist');
                }
                break;
            case 'paragraph':
                if (!ParagraphsType::load($type)) {
                    throw new Exception('Paragraph type ['.$type.'] does not exist');
                }
                break;
        }

        try {
            $fieldSettings = $this->prepareFieldSettings($options);
        } catch (Exception $exception) {
            Log::error($exception->getMessage());

            return false; // Fail to prepare field settings
        }

        // Try to get field storage
        $drupalFieldName = $fieldName;
        if (!in_array($drupalFieldName, ['status', 'body', 'author'])) {
            $drupalFieldName = 'field_'.$drupalFieldName;
        }

        $fieldStorage = FieldStorageConfig::loadByName($entityTypeId, $drupalFieldName);
        $fieldStorageSettings = $this->prepareFieldStorageSettings($options);
        if (isset($fieldStorageSettings['allowed_values'])) {
            // Update storage settings for lists
            $fieldStorage->setSetting(
                'allowed_values',
                Arr::pluck($fieldStorageSettings['allowed_values'], 'label', 'value')
            )->save();
        }

        if ($fieldStorage === null) {
            throw new Exception('storage type ['.$fieldName.'] does not exists');
        }

        if (($fieldConfig = FieldConfig::loadByName(
            $entityTypeId,
            $type == 'users' ? 'user' : $type,
            $drupalFieldName
            )) === null) {
            throw new Exception('Field ['.$fieldName.'] does not exists');
        }

        // Check if node type $type is translatable
        $translatable = false;
        try {
            $contentTranslationManager = Drupal::service('content_translation.manager');
            $translatable = $contentTranslationManager->isEnabled(
                $entityTypeId,
                $type == 'users' ? 'user' : $type
            );
        } catch (Exception $exception) {
            Log::error('[ManageField::contentTypeUpdateField] failed to get translatable => '.$exception->getMessage());
        }

        $configLanguage = $fieldConfig->language()->getId();

        $languageManager = Drupal::languageManager();

        // Manage field config options change
        if (array_key_exists('default', $options)) {
            $fieldConfig->set('default_value', $options['default']);
        }
        $fieldConfig->set('label', $options['title'][$configLanguage] ?? $fieldName);
        $fieldConfig->set('description', $options['description'][$configLanguage] ?? '');
        $fieldConfig->set('required', $options['required'] ?? false);
        $fieldConfig->set('translatable', $translatable && $options['translatable']);
        $fieldConfig->setSettings($fieldSettings);

        $fieldConfig->save();

        // Third party settings
        foreach ($this->getFieldSettingsFromOptions($options) as $settingKey => $settingValue) {
            $fieldConfig->setThirdPartySetting('inside', $settingKey, $settingValue);
        }
        $fieldConfig->save();

        // Manage translations

        // build lang key
        $key = 'field.field.'.$entityTypeId.'.'.($type == 'users' ? 'user' : $type).'.'.$drupalFieldName;

        // Save label for each language
        foreach ($this->languages as $language) {
            if ($language == $configLanguage) {
                continue;
            }
            if (array_key_exists('title', $options) && array_key_exists($language, $options['title'])) {
                $languageManager->getLanguageConfigOverride($language, $key)->set(
                    'label',
                    $options['title'][$language]
                )->save();
            }
            if (array_key_exists('description', $options) && array_key_exists($language, $options['description'])) {
                $languageManager->getLanguageConfigOverride($language, $key)->set(
                    'description',
                    $options['description'][$language]
                )->save();
            }
        }

        // Save label for each language
        $key = 'field.storage.'.$entityTypeId.'.'.$drupalFieldName;

        foreach ($this->languages as $language) {
            $settings = $this->prepareFieldStorageSettings($options, $language);
            if (!empty($settings) && isset($settings['allowed_values'])) {
                $languageManager->getLanguageConfigOverride($language, $key)->set(
                    'settings.allowed_values',
                    array_values($settings['allowed_values'])
                )->save();
            }
        }
        $fieldConfig->save();

        // Manage form display !
        if ($options['widget'] && is_array($options['widget']) && array_key_exists('type', $options['widget'])) {
            $formDisplay = \Drupal::service('entity_display.repository')->getFormDisplay(
                $entityTypeId,
                $type == 'users' ? 'user' : $type
            );
            if (array_key_exists('start', $options['widget'])) {
                $weight += (int) $options['widget']['start'];
            }
            if ($weight === null && isset($options['widget']['weight'])) {
                $weight = $options['widget']['weight'];
            }
            $formDisplay->setComponent(
                $drupalFieldName,
                [
                    'type' => $this->getDrupalWidgetFromInside($options['widget']['type']),
                    'settings' => $options['widget']['settings'] ?? [],
                    'weight' => $weight,
                ]
            )->save();
            if ($entityTypeId === 'node') {
                $this->prepareSystemGroups($formDisplay, $type);
            }
            $this->attachFieldToGroup($formDisplay, $options, $type, $fieldName);

            // Hide ?
            if (isset($options['widget']['hidden']) && $options['widget']['hidden']) {
                $formDisplay->removeComponent($drupalFieldName);
                $formDisplay->save();
            }
        }

        $this->writeln(' <fg=green>✔</fg=green>');

        return true;
    }


    /**
     * @param $entityType
     * @param $type
     * @param $fieldName
     *
     * @throws EntityStorageException
     */
    public function contentTypeDeleteField($entityType, $type, $fieldName)
    {
        if (!in_array($fieldName, ['body'])) {
            $fieldName = 'field_'.$fieldName;
        }
        $fieldConfig = FieldConfig::loadByName($entityType, $type, $fieldName);
        if (!$fieldConfig) {
            return;
        }
        $this->write(
            '<info>Supression du champ <fg=blue>'.$fieldName.'</fg=blue> au type <fg=yellow>'.$type
            .'</fg=yellow></info>'
        );
        $storageDefinition = $fieldConfig->getFieldStorageDefinition();
        $fieldConfig->delete();
        if ($storageDefinition->isDeletable()) {
            $storageDefinition->delete();
        }
        $this->writeln(' <fg=green>✔</fg=green>');
    }


    /**
     * Update field $fieldName inside settings
     *
     * @param  string  $type  inside type
     * @param  string  $fieldName  inside fieldName
     * @param  array  $options  inside third party options
     *
     * @throws EntityStorageException|Exception
     */
    public function updateFieldOptions(string $type, string $fieldName, array $options)
    {
        $entityTypeId = 'node';
        $drupalType = $type;
        if ($type === 'users') {
            $drupalType = 'user';
            $entityTypeId = 'user';
        } elseif (Schema::isSectionType($type)) {
            $entityTypeId = 'paragraph';
        }

        if (($fieldConfig = FieldConfig::loadByName(
            $entityTypeId,
            $drupalType,
            $fieldName == 'body' ? 'body' : "field_$fieldName"
            )) === null) {
            throw new Exception('Field ['.$fieldName.'] in type ['.$drupalType.'] does not exists');
        }
        $actualThirdPartySettings = $fieldConfig->getThirdPartySettings('inside');
        $newThirdPartySettings = $this->getFieldSettingsFromOptions($options);
        foreach ($newThirdPartySettings as $settingKey => $settingValue) {
            if (is_array($settingValue)
                && isset($actualThirdPartySettings[$settingKey])
                && is_array($actualThirdPartySettings[$settingKey])) {
                $settingValue = array_merge($actualThirdPartySettings[$settingKey], $settingValue);
            }

            if ($settingKey === 'special' && array_key_exists('special', $options) && array_key_exists('image_styles', $options['special'])) {
                $settingValue['image_styles'] = [];

                foreach ($options['special']['image_styles'] as $style) {
                    $settingValue['image_styles'][$style] = $style;
                }
            }

            $fieldConfig->setThirdPartySetting('inside', $settingKey, $settingValue);
        }
        if (isset($options['required'])) {
            $fieldConfig->set('required', $options['required'] ?? false);
        }
        if (isset($options['translatable'])) {
            $fieldConfig->set('translatable', $options['translatable'] ?? false);
        }

        $fieldOptions = Schema::getFieldOptions($type, $fieldName);

        if (isset($options['target_type'])
            && is_array($options['target_type'])
            && $fieldOptions['type'] === 'section') {
            $fieldConfig->setSettings(
                array_merge(
                    $fieldConfig->getSettings(),
                    $this->prepareFieldSettings(
                        ['type' => self::SECTION_FIELD, 'settings' => ['target_type' => $options['target_type']]]
                    )
                )
            );
        }
        $fieldConfig->save();
    }


    /**
     * Prepare settings for third party settings from inside options
     *
     * @param  array  $options
     *
     * @return array
     */
    protected function getFieldSettingsFromOptions(array $options): array
    {
        $settings = [];
        foreach ($options as $settingKey => $value) {
            if (array_key_exists($settingKey, $this->fieldOptions)
                && (isset($this->fieldOptions[$settingKey]['setting']) && $this->fieldOptions[$settingKey]['setting'])
                && (gettype($value) == $this->fieldOptions[$settingKey]['type']
                    || $this->fieldOptions[$settingKey]['type'] == 'mixed')
            ) {
                if (isset($this->fieldOptions[$settingKey]['group'])) {
                    $group = $this->fieldOptions[$settingKey]['group'];
                    if (!array_key_exists($group, $settings)) {
                        $settings[$group] = [];
                    }
                    if ($settingKey !== 'image_styles') {
                        $settings[$group][$settingKey] = $value;
                    }
                } else {
                    $settings[$settingKey] = $value;
                }
            }
        }
        if (isset($options['type']) && $options['type'] === self::IMAGE_FIELD) {
            $dataImageStyle = $this->manageFieldImageStyles($options);
            $settings = array_merge($settings, $dataImageStyle);
        }

        return $settings;
    }

    /**
     * Change field title
     *
     * @param  string  $type
     * @param  string  $fieldName
     * @param  array  $titles
     * @param  string  $entityTypeId
     * @return bool
     * @throws EntityStorageException
     * @throws Exception
     */
    public function changeFieldTitle(
        string $type,
        string $fieldName,
        array $titles,
        string $entityTypeId = "node"
    ): bool {
        if ($type === 'users') {
            $entityTypeId = 'user';
        }

        switch ($entityTypeId) {
            case 'node':
                if (!NodeType::load($type)) {
                    throw new Exception('Node type ['.$type.'] does not exist');
                }
                break;
            case 'paragraph':
                if (!ParagraphsType::load($type)) {
                    throw new Exception('Paragraph type ['.$type.'] does not exist');
                }
                break;
        }

        if ($entityTypeId !== 'user' && in_array($fieldName, $this->drupalFields[$entityTypeId])
        ) {
            return false;
        }

        // Try to get field storage
        $drupalFieldName = $fieldName;
        if (!in_array($drupalFieldName, ['status', 'body', 'author', 'field_body'])) {
            $drupalFieldName = 'field_'.$drupalFieldName;
        }

        if (($fieldConfig = FieldConfig::loadByName(
            $entityTypeId,
            $type == 'users' ? 'user' : $type,
            $drupalFieldName
        )) === null) {
            throw new Exception('Field ['.$fieldName.'] does not exists');
        }

        $configLanguage = $fieldConfig->language()->getId();

        $languageManager = Drupal::languageManager();

        $fieldConfig->set('label', $titles[$configLanguage] ?? $fieldName);

        $fieldConfig->save();

        // Manage translations
        // build lang key
        $key = 'field.field.'.$entityTypeId.'.'.($type == 'users' ? 'user'
                : $type).'.'.$drupalFieldName;

        // Save label for each language
        foreach ($this->languages as $language) {
            if ($language == $configLanguage) {
                continue;
            }
            if (array_key_exists($language, $titles)
            ) {
                $languageManager->getLanguageConfigOverride($language, $key)->set('label', $titles[$language])->save();
            }
        }

        return true;
    }

    public function changeContentTypeTitle(
        string $type,
        array $titles,
        string $entityTypeId = "node"
    ): bool {
        if ($entityTypeId !== 'node') {
            throw new Exception('Invalid entity type [' . $entityTypeId . ']. Only node content types are supported.');
        }

        $contentType = NodeType::load($type);
        if ($contentType === null) {
            throw new Exception('Content type [' . $type . '] does not exist.');
        }

        $configLanguage = $contentType->language()->getId();
        $languageManager = \Drupal::languageManager();

        $contentType->set('name', $titles[$configLanguage] ?? $type);
        $contentType->setThirdPartySetting('inside', 'title', $titles);
        $contentType->save();

        $options = array_merge($contentType->getThirdPartySettings('inside'), ['name' => $type]);
        DB::table('inside_models')->where('class', type_to_class($type))->update([
            'options' => json_encode($options),
        ]);

        $key = 'node.type.' . $type;

        foreach ($this->languages as $language) {
            if ($language === $configLanguage) {
                continue;
            }

            if (array_key_exists($language, $titles)) {
                $languageManager->getLanguageConfigOverride($language, $key)->set('name', $titles[$language])->save();
            }
        }

        return true;
    }

    /**
     * Set the default image_styles
     *
     * @param $options
     *
     * @return array
     */
    protected function manageFieldImageStyles($options): array
    {
        if (array_key_exists('image_styles', $options) && !empty($options['image_styles'])) {
            $options['special']['image_styles'] =
                ImageStyles::whereIn('title', $options['image_styles'])->pluck('title', 'title')->toArray();

            return $options;
        }
        $options["special"]["image_styles"] = ImageStyles::pluck('title', 'title')->toArray();

        return $options;
    }

    /**
     * @throws Drupal\locale\StringStorageException
     * @throws InvalidPluginDefinitionException
     * @throws PluginNotFoundException
     * @throws Exception
     */
    public function changeFieldFormOptions(
        string $type,
        string $fieldName,
        bool $displayed,
        ?string $group = null
    ): bool {
        $entityType = $type === 'users' ? 'user' : 'node';
        $bundle = $type === 'users' ? 'user' : $type;

        $this->write(
            '<info>Modification du champ <fg=blue>'.$fieldName.'</fg=blue> au type <fg=yellow>'.$type
            .'</fg=yellow></info> => Affichage <fg=magenta>'.($displayed ? 'OUI' : 'NON')
            .'</fg=magenta> group <fg=cyan>'.$group.'</fg=cyan>'
        );

        // Load form
        $formDisplay = Drupal::entityTypeManager()->getStorage('entity_form_display')->load(
            $entityType.'.'.$bundle.'.default'
        );
        if ($formDisplay === null) {
            $this->writeln(
                '<warning>Pas de formulaire</warning>'
            );

            return false;
        }
        if (!Schema::hasField($type, $fieldName)) {
            return false;
        }

        $fieldOptions = Schema::getFieldOptions($type, $fieldName);
        $drupalField = array_flip($this->swappableNames);
        $drupalFieldName = $fieldName;
        if (array_key_exists($drupalFieldName, $drupalField)) {
            $drupalFieldName = $drupalField[$drupalFieldName];
        } else {
            $drupalFieldName = $this->getDrupalFieldNameFromInsideFieldName($drupalFieldName);
        }

        $component = $formDisplay->getComponent($drupalFieldName);

        if ($component && !$displayed) {
            $formDisplay->removeComponent($drupalFieldName);
        } elseif (!$component && $displayed) {
            $formDisplay->setComponent(
                $drupalFieldName,
                [
                    'type' => $this->getDrupalWidgetFromOptions($fieldOptions['widget']),
                ]
            )->save();
        }

        if ($group !== null) {
            $this->prepareSystemGroups($formDisplay, $type);
            $this->attachFieldToGroup($formDisplay, ['widget' => ['group' => $group]], $type, $fieldName);
        } else {
            $this->detachFieldFromAnyGroup($formDisplay, $fieldName);
        }

        try {
            $formDisplay->save();
        } catch (Exception $exception) {
            $this->writeln(' <fg=red>✘</fg=red>');

            return false;
        }
        $this->writeln(' <fg=green>✔</fg=green>');

        return true;
    }

    /**
     * Reorder fields
     *
     * @param  string  $type
     * @param  array  $fieldNames
     *
     * @return bool
     * @throws InvalidPluginDefinitionException
     * @throws PluginNotFoundException
     * @throws EntityStorageException
     */
    public function reorderFields(string $type, array $fieldNames): bool
    {
        $entityType = $type === 'users' ? 'user' : 'node';
        $bundle = $type === 'users' ? 'user' : $type;

        // Load form
        $formDisplay = Drupal::entityTypeManager()->getStorage('entity_form_display')->load(
            $entityType.'.'.$bundle.'.default'
        );
        if ($formDisplay === null) {
            return false;
        }
        $otherFieldNames = Arr::sort(
            array_filter(
                Schema::getFieldListing($type),
                function ($value) use ($fieldNames) {
                    return !in_array($value, $fieldNames);
                }
            ),
            function ($fieldName) use ($type) {
                $fieldOptions = Schema::getFieldOptions($type, $fieldName);

                return $fieldOptions['weight'] ?? null;
            }
        );

        $fieldNames = $fieldNames + $otherFieldNames;
        $weight = 0;
        foreach ($fieldNames as $fieldName) {
            $drupalField = array_flip($this->swappableNames);
            if (array_key_exists($fieldName, $drupalField)) {
                $fieldName = $drupalField[$fieldName];
            } else {
                $fieldName = $this->getDrupalFieldNameFromInsideFieldName($fieldName);
            }
            $component = $formDisplay->getComponent($fieldName);
            if (!$component) {
                continue;
            }
            $component['weight'] = $weight++;
            try {
                $formDisplay->setComponent($fieldName, $component)->save();
            } catch (Exception $exception) {
                return false;
            }
        }
        $formDisplay->save();

        return true;
    }

    /**
     * The idea is to reorganise field weight and get a free space after
     * field $field to add something
     *
     * @param  string  $type
     * @param  string  $field
     *
     * @return int free space
     * @throws Exception
     */
    public function getFreeSpaceAfter(string $type, string $field): int
    {
        $fields = Schema::getFieldListing($type);

        $orderedFields = collect();
        foreach ($fields as $fieldName) {
            $orderedFields[$fieldName] = Schema::getFieldOptions($type, $fieldName);
        }
        $orderedFields = $orderedFields->sortBy('weight');
        $weight = 1;
        $freeWeight = null;
        $formDisplay = \Drupal::service('entity_display.repository')->getFormDisplay($type == 'users' ? 'user' : 'node', $type == 'users' ? 'user' : $type);
        foreach ($orderedFields->keys() as $fieldName) {
            if ($fieldName == $field) {
                $freeWeight = $weight++;
            }
            $component = $formDisplay->getComponent('field_'.$fieldName);
            if ($component === null) {
                $component = $formDisplay->getComponent('field_'.$fieldName);
            }
            $component['weight'] = $weight++;
            $formDisplay->setComponent('field_'.$fieldName, $component)->save();
        }

        return $freeWeight ?? $weight;
    }


    /**
     * Get the drupal widget name from inside widget
     *
     * Valid drupal field type
     *
     * comment_default, entity_reference_revisions_autocomplete, file_generic, image_image, link_default,
     * menu_item_extras_view_mode_selector_select, path, text_textarea, text_textarea_with_summary, text_textfield,
     * entity_reference_paragraphs, paragraphs, datetime_timestamp, boolean_checkbox, email_default,
     * entity_reference_autocomplete_tags, entity_reference_autocomplete, language_select, number,
     * options_buttons, options_select, string_textarea, string_textfield, uri
     *
     * @param  int  $widget
     *
     * @return string
     * @throws Exception
     */
    protected function getDrupalWidgetFromInside(int $widget): string
    {
        if (array_key_exists($widget, $this->widgetMap)) {
            return $this->widgetMap[$widget];
        }
        throw new Exception('['.$widget.'] is not a valid inside widget');
    }

    /**
     * get Inside widget id from drupal widget name
     *
     * @param  string  $widget
     *
     * @return int
     * @throws Exception
     */
    protected function getInsideFromDrupalWidget(string $widget): int
    {
        if (in_array($widget, $this->widgetMap)) {
            return array_search($widget, $this->widgetMap);
        }
        throw new Exception('['.$widget.'] is not a valid inside widget');
    }

    /**
     * Get a drupal widget type from a widget option ( from inside_fields )
     *
     * @param  string  $widget
     *
     * @return string
     * @throws Exception
     */
    protected function getDrupalWidgetFromOptions(string $widget): string
    {
        switch ($widget) {
            case 'comments':
                return 'comment_default';
            case 'autocomplete':
                return 'entity_reference_autocomplete';
            case 'section':
                return 'entity_reference_paragraphs';
            case 'checkboxes':
            case 'onoff':
                return 'options_buttons';
            case 'select':
                return 'options_select';
            case 'image':
                return 'image_image';
            case 'file':
                return 'file_generic';
            case 'text':
                return 'string_textfield';
            case 'text_textarea_with_summary':
                return 'wysiwyg';
            case 'datetime':
                return 'datetime_timestamp';
            case 'email':
                return 'email_default';
        }
        throw new Exception('['.$widget.'] is not a valid inside widget option');
    }

    /**
     * Get drupal field storage settings
     *
     * @param  string  $type
     * @param  string  $fieldName
     * @param  string  $domain
     * @return array|array[]|false|int[]
     * @throws Exception
     */
    public function getDrupalStorageSettings(string $type, string $fieldName, string $domain = 'node')
    {
        if (in_array($fieldName, $this->drupalFields['node'])) {
            return false;
        }

        $fieldStorage = FieldStorageConfig::loadByName(
            $type == 'users' ? 'user' : $domain,
            $fieldName == 'body' ? "body" : "field_$fieldName"
        );
        if ($fieldStorage === null) {
            throw new Exception('storage type ['.$fieldName.'] does not exists');
        }

        if (($config = FieldConfig::loadByName(
            ($type == 'users') ? 'user' : $domain,
            ($type == 'users') ? 'user' : $type,
            $fieldName == 'body' ? "body" : "field_$fieldName"
        )) === null) {
            throw new Exception('field type ['.$fieldName.'] does not exists in content type ['.$type.']');
        }
        $settings = array_merge(
            [
                'type' => $this->getInsideTypeFromDrupal($fieldStorage->get('type')),
                'settings' => [],
            ],
            [
                'settings' => $fieldStorage->getSettings(),
            ]
        );
        $fieldSettings = $config->getSettings();
        if (isset($fieldSettings['handler_settings']['target_bundles'])
            && $fieldSettings['handler_settings']['target_bundles'] !== null
        ) {
            // References
            $settings['settings']['target_type'] = array_values($fieldSettings['handler_settings']['target_bundles']);
        } elseif (isset($fieldSettings['handler_settings']['target_bundles_drag_drop'])
            && $fieldSettings['handler_settings']['target_bundles_drag_drop'] !== null
        ) {
            // Paragraphs
            $settings['settings']['target_type'] =
                array_keys($fieldSettings['handler_settings']['target_bundles_drag_drop']);
        }

        if (($cardinality = $fieldStorage->get('cardinality')) !== null) {
            $settings['settings']['cardinality'] = $cardinality;
        }

        return $settings;
    }

    /**
     * @throws EntityStorageException
     * @throws Exception
     */
    public function switchCardinalityFromUnaryToMultiple(string $type, string $fieldName, bool $force = false): bool
    {
        $entityTypeId = 'node';
        if ($type === 'users') {
            $entityTypeId = 'user';
        } elseif (Schema::isSectionType($type)) {
            $entityTypeId = 'paragraph';
        }

        if (FieldConfig::loadByName(
            $entityTypeId,
            $type == 'users' ? 'user' : $type,
            'field_'.$fieldName
        ) === null) {
            return false;
        }
        switch ($entityTypeId) {
            case 'node':
                if (!NodeType::load($type)) {
                    throw new Exception('Node type ['.$type.'] does not exist');
                }
                break;
            case 'paragraph':
                if (!ParagraphsType::load($type)) {
                    throw new Exception('Paragraph type ['.$type.'] does not exist');
                }
                break;
        }

        if ($entityTypeId !== 'user' && in_array($fieldName, $this->drupalFields[$entityTypeId])) {
            return false;
        }

        if (!in_array($fieldName, ['status', 'body', 'author'])) {
            $fieldName = 'field_'.$fieldName;
        }

        $fieldStorage = FieldStorageConfig::loadByName($entityTypeId, $fieldName);
        if (!$fieldStorage) {
            return false;
        }
        $cardinality = $fieldStorage->get('cardinality');
        if (!$force && $cardinality === -1) {
            return true;
        }

        $fieldStorage->setCardinality(-1);
        return $fieldStorage->save() === SAVED_UPDATED;
    }


    /**
     *
     * @param  array  $options
     * @param  string|null  $langCode
     * @return array
     * @throws Exception
     */
    protected function prepareFieldStorageSettings(array $options, string $langCode = null): array
    {
        $type = $options['type'] ?? self::TEXT_FIELD;
        $settings = $options['settings'] ?? [];
        $langCode = $langCode ?? $this->defaultLanguage;
        switch ($type) {
            case self::REFERENCE_FIELD:
                return [
                    'settings' => [
                        'target_type' => (($settings['target_type'] == 'users') ? 'user' : 'node'),
                    ],
                    'cardinality' => ($settings['cardinality']
                        != FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED) ? $settings['cardinality']
                        : FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED,
                ];
            case self::SECTION_FIELD:
                return [
                    'settings' => [
                        'target_type' => 'paragraph',
                    ],
                    'cardinality' => ($settings['cardinality']
                        != FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED) ? $settings['cardinality']
                        : FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED,
                ];
            case self::DECIMAL_FIELD:
                return [
                    'cardinality' => 1,
                    'precision' => $settings['precision'] ?? 10,
                    'scale' => $settings['scale'] ?? 2,
                ];
            case self::TEXT_FIELD:
            case self::TEXTAREA_FIELD:
            case self::WYSIWYG_FIELD:
            case self::IMAGE_FIELD:
            case self::FILE_FIELD:
            case self::FLOAT_FIELD:
            case self::INTEGER_FIELD:
            case self::TIMESTAMP_FIELD:
            case self::URI_FIELD:
            case self::UUID_FIELD:
            case self::BOOLEAN_FIELD:
            case self::LONGTEXT_FIELD:
            case self::LINK_FIELD:
                return ['cardinality' => 1];
            case self::COMMENT_FIELD:
                return ['comment_type' => $settings['comment_type'] ?? 'comments',];
            case self::LIST_FLOAT_FIELD:
            case self::LIST_INTEGER_FIELD:
                $fieldSettings = [
                    'cardinality' => 1,
                    'allowed_values' => [],
                ];
                foreach ($settings['allowed_values'] as $value) {
                    $fieldSettings['allowed_values'][] = [
                        'value' => $value,
                        'label' => $value,
                    ];
                }

                return $fieldSettings;
                break;
            case self::LIST_STRING_FIELD:
                $fieldSettings = [
                    'cardinality' => 1,
                    'allowed_values' => [],
                ];
                foreach ($settings['allowed_values'] as $key => $value) {
                    $fieldSettings['allowed_values'][$key] = [
                        'value' => $key,
                        'label' => $value[$langCode],
                    ];
                }

                return $fieldSettings;
        }
        throw new Exception('Field type ['.$options['type'].'] not known');
    }

    /**
     * Prepare field options
     *
     *
     * @param  array  $options
     *
     * @return array
     * @throws Exception
     */
    protected function prepareFieldSettings(array $options): array
    {
        $type = $options['type'] ?? self::TEXT_FIELD;
        $settings = $options['settings'] ?? [];
        switch ($type) {
            case self::REFERENCE_FIELD:
                // Check target exists !
                if (!array_key_exists('target_type', $settings)) {
                    throw new Exception('Target type is not set');
                }
                $targets = $settings['target_type'];
                if (!is_array($targets)) {
                    $targets = [$targets];
                }
                if (empty($targets)) {
                    return [];
                }

                if (count($targets) > 1 && array_search('users', $targets) !== false) {
                    throw new Exception('You cannot target users with other nodes');
                }
                // Special case for user
                if ($targets[0] == 'users') {
                    return [
                        'handler' => 'default:user',
                        'handler_settings' => [
                            'include_anonymous' => false,
                            'target_bundles' => null,
                            'auto_create' => false,
                        ],
                        'target_type' => 'user',
                    ];
                }

                $bundles = [];
                foreach ($targets as $target) {
                    if ($target != 'user' && !NodeType::load($target)) {
                        throw new Exception('Target type ['.$target.'] does not exist');
                    }

                    $bundles[$target] = $target;
                }

                return [
                    'handler' => 'default:node',
                    'handler_settings' => [
                        'target_bundles' => $bundles,
                    ],
                    'target_type' => 'node',
                ];
            case self::SECTION_FIELD:
                // Check target exists !
                if (!array_key_exists('target_type', $settings)) {
                    throw new Exception('Target type is not set');
                }

                foreach ($settings['target_type'] as $target) {
                    if (!ParagraphsType::load($target)) {
                        throw new Exception('Target type ['.$target.'] does not exist');
                    }
                }

                $bundles = [];
                $i = 1;
                foreach ($settings['target_type'] as $target) {
                    $bundles[$target] = [
                        'weight' => $i++,
                        'enabled' => true,
                    ];
                }

                return [
                    'handler' => 'default:paragraph',
                    'handler_settings' => [
                        'target_bundles_drag_drop' => $bundles,
                    ],
                ];
            case self::TEXT_FIELD:
                $fieldSettings = [
                    'size' => $settings['size'] ?? 60,
                    'max_length' => $settings['max_length'] ?? 191,
                ];
                if (isset($options['widget'])) {
                    switch ($options['widget']) {
                        case self::INSIDE_PHONE_WIDGET:
                            $fieldSettings['short_number_length'] = $settings['short_number_length'] ?? 0;
                            break;
                    }
                }
                return $fieldSettings;
            case self::TEXTAREA_FIELD:
                return ['max_length' => $settings['max_length'] ?? 191];
            case self::WYSIWYG_FIELD:
            case self::TIMESTAMP_FIELD:
            case self::LONGTEXT_FIELD:
            case self::LINK_FIELD:
                return [
                    'title' => $settings['title'] ?? 1, //Link text is not mandatary by default
                    'link_type' => $settings['link_type'] ?? 17 //Links can be internal and external by default
                ];
            case self::URI_FIELD:
            case self::UUID_FIELD:
            case self::LIST_STRING_FIELD:
            case self::LIST_INTEGER_FIELD:
            case self::LIST_FLOAT_FIELD:
            case self::BOOLEAN_FIELD:
            case self::COMMENT_FIELD:
                return [];
            case self::IMAGE_FIELD:
                return [
                    'file_directory' => $settings['file_directory'] ?? '[date:custom:Y]-[date:custom:m]',
                    'file_extensions' => $settings['file_extensions'] ?? 'png gif jpg jpeg',
                    'max_filesize' => $settings['max_filesize'] ?? '',
                    'mimetypes' => $settings['mimetypes'] ?? '',
                    'max_resolution' => $settings['max_resolution'] ?? '',
                    'min_resolution' => $settings['min_resolution'] ?? '',
                    'alt_field' => false,
                    'alt_field_required' => false,
                    'title_field' => false,
                    'title_field_required' => false,
                    'handler' => 'default:file',
                    'handler_settings' => [],
                    'uri_scheme' => 'public',
                    'target_type' => 'file',
                    'display_field' => false,
                    'display_default' => false,
                    'default_image' => $settings['default_image'] ?? [
                            'uuid' => '',
                            'alt' => '',
                            'title' => '',
                            'width' => null,
                            'height' => null,
                        ],
                ];
            case self::FLOAT_FIELD:
            case self::DECIMAL_FIELD:
            case self::INTEGER_FIELD:
                return [
                    'min' => $settings['min'] ?? null,
                    'max' => $settings['max'] ?? null,
                    'prefix' => $settings['prefix'] ?? '',
                    'suffix' => $settings['suffix'] ?? '',
                ];
            case self::FILE_FIELD:
                return [
                    'file_directory' => $settings['file_directory'] ?? '[date:custom:Y]-[date:custom:m]',
                    'mimetypes' => $settings['mimetypes'] ?? '',
                    'file_extensions' => $settings['file_extensions'] ??
                        'doc docx odt txt pdf xls xlsx csv ppt pptx ai jpeg jpg gif png svg tif mp4 avi mov webm zip',
                    'max_filesize' => $settings['max_filesize'] ?? '',
                    'description_field' => false,
                    'handler' => 'default:file',
                    'handler_settings' => [],
                    'display_field' => false,
                    'display_default' => false,
                    'uri_scheme' => 'public',
                    'target_type' => 'file',
                ];
        }
        throw new Exception('Field type ['.$options['type'].'] not known');
    }


    /**
     * Custom drupal field name are prefixed by field_
     *
     * This function manage prefixes so that we don't need to care about it
     *
     * @param  string  $fieldName
     *
     * @return string
     */
    public function getDrupalFieldNameFromInsideFieldName(string $fieldName): string
    {
        if (in_array($fieldName, $this->drupalFields['node'])) {
            return $fieldName;
        }

        return 'field_'.$fieldName;
    }

    /**
     * Get inside fieldName from drupal one
     *
     * @param  string  $fieldName
     *
     * @return string
     */
    public function getInsideFieldNameFromDrupalFieldName(string $fieldName): string
    {
        return Str::after($fieldName, 'field_');
    }

    /**
     * Get widget option of field name of type
     *
     * @param  string  $type
     * @param  string  $name
     *
     * @return array
     * @throws Exception
     */
    public function getFormWidgetOptions(string $type, string $name): array
    {
        $name = $this->getDrupalFieldNameFromInsideFieldName($name);
        $formDisplay = \Drupal::service('entity_display.repository')->getFormDisplay(
            $type == 'users' ? 'user' : 'node',
            $type == 'users' ? 'user' : $type
        );
        $component = $formDisplay->getComponent($name);
        if ($component === null) {
            return [
                'hidden' => true,
            ];
        }

        return [
            'type' => $this->getInsideFromDrupalWidget($component['type']),
            'settings' => $component['settings'],
            'weight' => $component['weight'],
        ];
    }
}
