<?php

namespace Inside\Host\Bridge\Traits;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Config\Entity\ConfigEntityBundleBase;
use Drupal\Core\Entity\Display\EntityFormDisplayInterface;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\language\Entity\ContentLanguageSettings;
use Drupal\locale\SourceString;
use Drupal\locale\StringStorageException;
use Drupal\node\Entity\NodeType;
use Drupal\paragraphs\Entity\ParagraphsType;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Content\Facades\DynamicClass;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Contents\Users;
use Inside\Content\Models\Model;
use Inside\Host\Exceptions\ColumnAlreadyExistsException;
use Inside\Host\Exceptions\TableAlreadyExistException;
use Inside\Support\DrupalConfig;
use Throwable;

/**
 * Manage Content Type and Drupal Entities
 */
trait ManageContentType
{
    protected array $allowedContentTypeOptions = [
        "searchable"         => 'boolean',
        "aliasable"          => 'boolean',
        "global_searchable"  => 'boolean',
        "permissible"        => 'boolean',
        "categorizable"      => 'boolean',
        "translatable"       => 'boolean',
        'search_users_name'  => 'boolean',
        'search_users_email' => 'boolean',
        'listing_type'       => 'string',
        'notifications'      => 'string',
        'deletion_strategy'  => 'string',
    ];


    /**
     * Update content type inside settings
     *
     * @param  string  $type  inside type
     * @param  array  $options  inside third party options
     *
     * @throws EntityStorageException
     */
    public function updateContentTypeOptions(string $type, array $options): void
    {
        $entityTypeId = 'node';
        $entityType = null;
        if ($type === 'users') {
            $this->updateUserTypeOptions($options);
            return;
        } elseif (Schema::isSectionType($type)) {
            $entityTypeId = 'paragraph';
        }

        // Check content type exists in drupal
        if ($entityTypeId === 'node' && ($entityType = NodeType::load($type)) === null) {
            throw new ModelNotFoundException;
        } elseif ($entityTypeId === 'paragraph' && ($entityType = ParagraphsType::load($type)) === null) {
            throw new ModelNotFoundException;
        }

        $this->updateDrupalEntitySettings($entityType, $options);
    }

    /**
     * Update specific user entity options
     *
     * @param  array  $options
     */
    protected function updateUserTypeOptions(array $options): void
    {
        $config = DrupalConfig::where('name', 'inside.user')->first();
        $config->config = $options;
        $config->save();

        DB::table('cache_config')->truncate();

        $model = Model::where('class', Users::class)->first();
        if ($model) {
            $modelOptions = $model->options;
            foreach ($options as $settingKey => $settingValue) {
                $modelOptions[$settingKey] = $settingValue;
            }
            $model->options = $modelOptions;
            $model->save();
        }
    }

    /**
     * Update a settings for a node type
     *
     * @param  NodeType  $nodeType
     * @param  array  $settings
     *
     * @throws EntityStorageException
     */
    protected function updateDrupalEntitySettings(ConfigEntityBundleBase $nodeType, array $settings): void
    {
        foreach ($this->allowedContentTypeOptions as $settingKey => $settingType) {
            if (array_key_exists($settingKey, $settings)
                && ((gettype($settings[$settingKey]) == $settingType)
                    || $settingType == 'mixed')
            ) {
                $nodeType->setThirdPartySetting('inside', $settingKey, $settings[$settingKey]);
            }
        }

        $nodeType->save();
    }

    /**
     * Create or Update a contentType
     */
    public function contentTypeUpdateOrCreate(
        string $type, array $options, array $fields = [], array $groups = [], string $domain = 'contents'
    ): bool {
        $entityTypeId = $type === 'users' ? 'user' : ($domain === 'contents' ? 'node' : 'paragraph');

        if ($type == 'users' || NodeType::load($type) || ParagraphsType::load($type)) {
            try {
                return $this->contentTypeUpdate($type, $options, $fields, $groups, $entityTypeId);
            } catch (EntityStorageException) {
                return false;
            }
        }

        try {
            return $this->contentTypeCreate($type, $options, $fields, $groups, $entityTypeId);
        } catch (EntityStorageException|StringStorageException|TableAlreadyExistException) {
            return false;
        }
    }

    /**
     * Content type creation ($type content type should not exist )
     *
     * @throws TableAlreadyExistException|StringStorageException|ColumnAlreadyExistsException|EntityStorageException
     */
    public function contentTypeCreate(
        string $type, array $options, array $fields = [], array $groups = [], string $entityTypeId = 'node'
    ): bool {
        // fallback on users
        if ($type === 'users') {
            $entityTypeId = 'user';
        }

        // Check content type does not already exist
        if ($entityTypeId === 'node' && NodeType::load($type)) {
            throw TableAlreadyExistException::named($type);
        } elseif ($entityTypeId === 'paragraph' && ParagraphsType::load($type)) {
            throw TableAlreadyExistException::named($type);
        }

        $this->write('<info>Création du type de contenu <fg=blue>' . $type . '</fg=blue></info>');

        $needToDisableRebuild = DynamicClass::isRebuildAutoLoadNeeded();
        if ($needToDisableRebuild) {
            DynamicClass::disableRebuildAutoLoad();
        }

        if ($entityTypeId !== 'user') {
            if ($entityTypeId === 'node') {
                $entityType = NodeType::create(
                    [
                        'type'     => $type,
                        'name'     => $options['name'] ?? $type,
                        'langcode' => $this->defaultLanguage,
                    ]
                );
            } else {
                $entityType = ParagraphsType::create(
                    [
                        'id'    => $type,
                        'label' => $options['title'][$this->defaultLanguage],
                    ]
                );
            }

            // Settings
            $this->updateDrupalEntitySettings($entityType, $options);

            // Manage language
            if (count(($this->languages)) > 1 && array_key_exists('translatable', $options)
                && (gettype($options['translatable']) == 'boolean')
                && $options['translatable']
            ) {
                // Enable translation
                $contentTranslationManager = \Drupal::service('content_translation.manager');
                $contentTranslationManager->setEnabled($entityTypeId, $entityType->id(), true);

                $config = ContentLanguageSettings::loadByEntityTypeBundle($entityTypeId, $entityType->id());
                $config->setDefaultLangcode($this->defaultLanguage)->setLanguageAlterable(true)->save();

                // Add and set config translations
            }

            $this->setContentTypeName($entityType, $options);
        }

        // Prepare form
        $form = \Drupal::service('entity_display.repository')->getFormDisplay($entityTypeId, $type === 'users' ? 'user' : $type);

        // Prepare groups
        if (empty($groups) && $entityTypeId === 'node') {
            $this->prepareSystemGroups($form, $type);
        }

        // TODO manage multilingue on groups
        foreach ($groups as $group) {
            field_group_group_save((object)$group, $form);
        }

        if ($entityTypeId === 'node') {
            // Hide fields
            $toHide = ['translation', 'sticky', 'path', 'promote', 'langcode'];

            $form = \Drupal::service('entity_display.repository')->getFormDisplay('node', $type);
            foreach ($toHide as $field) {
                $form->removeComponent($field);
            }
            $form->save();
        }

        $this->writeln(' <fg=green>✔</fg=green>');
        // Manage Fields
        $titleComponent = $form->getComponent('title');
        $weight         = $titleComponent['weight'];
        foreach ($fields as $options) {
            if (isset($options['name'])) {
                $this->contentTypeCreateField($type, $options['name'], $options, ++$weight, $entityTypeId);
            }
        }
        if ($needToDisableRebuild) {
            DynamicClass::enableRebuildAutoLoad();
            DynamicClass::rebuildAutoLoad();
        }

        return true;
    }

    /**
     * Update a content type
     *
     * @throws Exception|EntityStorageException|ModelNotFoundException
     */
    public function contentTypeUpdate(
        string $type, array $options = [], array $fields = [], array $groups = [], string $entityTypeId = 'node'
    ): bool {
        $entityType = null;
        // fallback on user
        if ($type === 'users') {
            $entityTypeId = 'user';
        }

        // Check content type exists
        if ($entityTypeId === 'node' && ($entityType = NodeType::load($type)) === null) {
            throw new ModelNotFoundException;
        } elseif ($entityTypeId === 'paragraph' && ($entityType = ParagraphsType::load($type)) === null) {
            throw new ModelNotFoundException;
        }
        if ($entityTypeId !== 'user' && $entityType === null) {
            throw new ModelNotFoundException;
        }

        $this->write('<info>Mise-à-jour du type de contenu <fg=blue>' . $type . '</fg=blue></info>');

        DynamicClass::disableRebuildAutoLoad();

        $this->defaultLanguage = $this->languages[0];

        if ($entityTypeId !== 'user') {
            // Settings
            $this->updateDrupalEntitySettings($entityType, $options);

            $config = ContentLanguageSettings::loadByEntityTypeBundle($entityTypeId, $entityType->id());
            $translatableDrupalSide = $config->isLanguageAlterable();
            $translatableThirdPartySide = $config->getThirdPartySetting('content_translation', 'enabled', false);

            // Manage language
            if ($translatableDrupalSide && array_key_exists('translatable', $options) && !$options['translatable']) {
                throw new Exception('[BridgeContentType] does not support to disable "translatable" on a Content Type');
            }

            if (count(($this->languages)) > 1
                && ! $translatableThirdPartySide
                && array_key_exists('translatable', $options)
                && (gettype($options['translatable']) == 'boolean')
                && $options['translatable']
            ) {
                // This content type is not translatable from third party but allowed to be from drupal and we wan't to set it translatable
                // Enable translation
                $contentTranslationManager = \Drupal::service('content_translation.manager');
                $contentTranslationManager->setEnabled($entityTypeId, $entityType->id(), true);

                $config = ContentLanguageSettings::loadByEntityTypeBundle($entityTypeId, $entityType->id());
                $config->setDefaultLangcode($this->defaultLanguage)->setLanguageAlterable(true)->save();
            }

            // Update config translations
            $this->setContentTypeName($entityType, $options);
        }
        $this->writeln(' <fg=green>✔</fg=green>');

        // Manage Fields
        $form           = \Drupal::service('entity_display.repository')->getFormDisplay($entityTypeId, $type);
        $titleComponent = $form->getComponent('title');
        $weight         = $titleComponent['weight'];

        // Prepare groups
        if (empty($groups) && $entityTypeId === 'node') {
          $this->prepareSystemGroups($form, $type);
        }

        // TODO manage multilingue on groups
        foreach ($groups as $group) {
          field_group_group_save((object)$group, $form);
        }

        foreach ($fields as $options) {
            if (isset($options['name'])) {
                $this->contentTypeCreateOrUpdateField($type, $options['name'], $options, ++$weight, $entityTypeId);
            }
        }
        DynamicClass::enableRebuildAutoLoad();
        DynamicClass::rebuildAutoLoad();

        return true;
    }

    /**
     * Delete content type ( it must be empty of content )
     *
     * @throws EntityStorageException
     */
    public function contentTypeDelete(string $type, string $domain = 'contents'): bool
    {
        $fields = Arr::except(
            Schema::getFieldListing($type),
            $this->drupalFields[$domain === 'contents' ? 'node' : 'paragraph']
        );
        foreach ($fields as $field) {
            $this->contentTypeDeleteField('node', $type, $field);
        }

        if ($domain === 'contents') {
            $entityType = NodeType::load($type);
        } else {
            $entityType = ParagraphsType::load($type);
        }

        if ($entityType !== null) {
            try {
                $entityType->delete();
                if($domain === 'contents'){
                    Model::query()->where('class', type_to_class($type))->delete();
                }
            } catch (Throwable $e) {
                Log::error('[BridgeContentType] Unable to delete content type '.$type.' : '.$e->getMessage());

                return false;
            }
        }

        return true;
    }


    /**
     * Set content Type name ( in all known languages )
     *
     * @throws EntityStorageException
     */
    protected function setContentTypeName(ConfigEntityBundleBase $entityType, array $options, string $entityTypeId = 'node'): void
    {
        foreach ($this->languages as $language) {
            if (array_key_exists('title', $options) && array_key_exists($language, $options['title'])) {
                if ($language == $this->defaultLanguage && $entityType->get('name') != $options['title'][$language]) {
                    $entityType->set('name', $options['title'][$language])->save();
                }
                $entityTypeTranslation = \Drupal::languageManager()->getLanguageConfigOverride($language,
                    $entityTypeId . '.type.' . $entityType->id()
                );
                $entityTypeTranslation->set('name', $options['title'][$language])->save();
            }
        }
    }



    /**
     * @param  string  $type
     * @param  int  $drupalType
     * @return array<string, string>
     * @throws Exception
     */
    protected function getDrupalTypeAndBundle(string $type, int $drupalType = self::NODE_TYPE): array
    {
        switch ($drupalType) {
            case self::NODE_TYPE:
                return ['node', $type];
            case self::USER_TYPE:
                return ['user', 'user'];
            case self::PARAGRAPH_TYPE:
                return ['paragraph', $type];
        }
        throw new Exception('Drupal Type ['.$drupalType.'] does not exists');
    }
}
