<?php

namespace Inside\Host\Bridge;

use Exception;
use Inside\Host\Bridge\Traits\ManageContentType;
use Inside\Host\Bridge\Traits\ManageField;
use Inside\Host\Bridge\Traits\ManageFieldGroup;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * Bridge for content creation
 *
 * @category  Class
 * @package   Inside\Host\Bridge
 * @author    Maecia <technique@maecia.com>
 * @link      https://www.maecia.com/
 */
class BridgeContentType extends Bridge
{
    use ManageContentType, ManageField, ManageFieldGroup;

    protected ?OutputInterface $output = null;

    /**
     * Valid plugin IDs for Drupal\Core\Field\WidgetPluginManager are: comment_default,
     * entity_reference_revisions_autocomplete, file_generic, image_image, link_default,
     * menu_item_extras_view_mode_selector_select, path, text_textarea, text_textarea_with_summary, text_textfield,
     * entity_reference_paragraphs, paragraphs, datetime_timestamp, boolean_checkbox, email_default,
     * entity_reference_autocomplete_tags, entity_reference_autocomplete, language_select, number, options_buttons,
     * options_select, string_textarea, string_textfield, uri
     */

    /**
     * Inside field types
     */
    const REFERENCE_FIELD = 1;

    const TEXT_FIELD = 2;

    const TEXTAREA_FIELD = 3;

    const WYSIWYG_FIELD = 4;

    const TIMESTAMP_FIELD = 5;

    const URI_FIELD = 6;

    const UUID_FIELD = 7;

    const SECTION_FIELD = 8;

    const IMAGE_FIELD = 9;

    const FILE_FIELD = 10;

    const COMMENT_FIELD = 11;

    const LIST_STRING_FIELD = 12;

    const BOOLEAN_FIELD = 13;

    const LONGTEXT_FIELD = 14;

    const FLOAT_FIELD = 15;

    const INTEGER_FIELD = 16;

    const LINK_FIELD = 17;

    const LIST_INTEGER_FIELD = 18;

    const DECIMAL_FIELD = 19;

    const LIST_FLOAT_FIELD = 20;

    const NODE_TYPE = 1;

    const USER_TYPE = 2;

    const PARAGRAPH_TYPE = 3;

    protected array $onlyDrupalFieldNames = [
        'access',
        'account',
        'behavior_settings',
        'bundle',
        'cid',
        'comment_type',
        'content_translation_changed',
        'content_translation_created',
        'content_translation_outdated',
        'content_translation_source',
        'content_translation_status',
        'content_translation_uid',
        'default_langcode',
        'description',
        'entity_id',
        'entity_type',
        'expanded',
        'external',
        'field_name',
        'homepage',
        'hostname',
        'id',
        'init',
        'login',
        'mail',
        'menu_name',
        'name',
        'nid',
        'parent',
        'parent_field_name',
        'path',
        'pid',
        'preferred_admin_langcode',
        'preferred_langcode',
        'promote',
        'rediscover',
        'revision_default',
        'revision_id',
        'revision_log',
        'revision_timestamp',
        'revision_translation_affected',
        'revision_uid',
        'roles',
        'sticky',
        'subject',
        'thread',
        'timezone',
        'translation',
        'type',
        'uid',
        'uuid',
        'vid',
        'view_mode',
    ];

    protected array $drupalWidgetMap = [
        'comment_default' => 'comments',
        'taxonomy_autocomplete' => 'autocomplete',
        'entity_reference_autocomplete_tags' => 'autocomplete',
        'entity_reference_autocomplete' => 'autocomplete',
        'entity_reference_revisions' => 'section',
        'entity_reference_paragraphs' => 'section',
        'options_buttons' => 'checkboxes',
        'options_select' => 'select',
        'options_onoff' => 'onoff',
        'boolean_checkbox' => 'onoff',
        'image_image' => 'image',
        'file_generic' => 'file',
        'string_textfield' => 'text',
        'string_textarea' => 'textarea',
        'text_textfield' => 'text',
        'text_textarea' => 'wysiwyg',
        'text_textarea_with_summary' => 'wysiwyg',
        'datetime_timestamp' => 'datetime',
        'inside_timestamp_field_widget' => 'inside_timestamp_field_widget',
        'inside_link_field_widget' => 'inside_link_field_widget',
        'inside_color_picker_field_widget' => 'color_picker',
        'inside_icon_picker_field_widget' => 'icon_picker',
        'inside_phone_field_widget' => 'phone',
        'list_string' => 'select',
        'list_float' => 'select',
        'list_integer' => 'select',
        'email_default' => 'email',
    ];

    protected array $typeMap = [
        self::REFERENCE_FIELD => 'entity_reference',
        self::TEXT_FIELD => 'string',
        self::TEXTAREA_FIELD => 'string_long',
        self::WYSIWYG_FIELD => 'text_with_summary',
        self::TIMESTAMP_FIELD => 'timestamp',
        self::URI_FIELD => 'uri',
        self::UUID_FIELD => 'uuid',
        self::SECTION_FIELD => 'entity_reference_revisions',
        self::IMAGE_FIELD => 'image',
        self::FILE_FIELD => 'file',
        self::COMMENT_FIELD => 'comment',
        self::LIST_STRING_FIELD => 'list_string',
        self::LIST_INTEGER_FIELD => 'list_integer',
        self::LIST_FLOAT_FIELD => 'list_float',
        self::BOOLEAN_FIELD => 'boolean',
        self::LONGTEXT_FIELD => 'text_long',
        self::FLOAT_FIELD => 'float',
        self::INTEGER_FIELD => 'integer',
        self::DECIMAL_FIELD => 'decimal',
        self::LINK_FIELD => 'link',
    ];

    /**
     * Drupal Widgets
     */
    const SELECT_WIDGET = 101;

    const TEXT_WIDGET = 102;

    const LONGTEXT_WIDGET = 103;

    const SECTION_WIDGET = 104;

    const WYSIWYG_WIDGET = 105;

    const IMAGE_WIDGET = 106;

    const FILE_WIDGET = 107;

    const BOOLEAN_CHECKBOX_WIDGET = 108;

    const TIMESTAMP_WIDGET = 109;

    const AUTOCOMPLETE_WIDGET = 110;

    const NUMBER_WIDGET = 111;

    const OPTIONS_BUTTONS_WIDGET = 112;

    const STRING_WIDGET = 113;

    const INSIDE_TIMESTAMP_WIDGET = 114;

    const INSIDE_LINK_WIDGET = 115;

    const INSIDE_COLOR_PICKER_WIDGET = 116;

    const INSIDE_ICON_PICKER_WIDGET = 117;

    const INSIDE_PHONE_WIDGET = 118;

    const DRUPAL_LINK_WIDGET = 119;

    protected array $widgetMap = [
        self::SELECT_WIDGET => 'options_select',
        self::AUTOCOMPLETE_WIDGET => 'entity_reference_autocomplete',
        self::TEXT_WIDGET => 'text_textfield',
        self::STRING_WIDGET => 'string_textfield',
        self::LONGTEXT_WIDGET => 'string_textarea',
        self::NUMBER_WIDGET => 'number',
        self::SECTION_WIDGET => 'entity_reference_paragraphs',
        self::WYSIWYG_WIDGET => 'text_textarea_with_summary',
        self::IMAGE_WIDGET => 'image_image',
        self::FILE_WIDGET => 'file_generic',
        self::BOOLEAN_CHECKBOX_WIDGET => 'boolean_checkbox',
        self::TIMESTAMP_WIDGET => 'datetime_timestamp',
        self::INSIDE_TIMESTAMP_WIDGET => 'inside_timestamp_field_widget',
        self::INSIDE_LINK_WIDGET => 'inside_link_field_widget',
        self::INSIDE_PHONE_WIDGET => 'inside_phone_field_widget',
        self::INSIDE_COLOR_PICKER_WIDGET => 'inside_color_picker_field_widget',
        self::INSIDE_ICON_PICKER_WIDGET => 'inside_icon_picker_field_widget',
        self::OPTIONS_BUTTONS_WIDGET => 'options_buttons',
        self::DRUPAL_LINK_WIDGET => 'link_default',
    ];

    /**
     * Groups form name
     */
    const GROUP_PUBLISHING_OPTIONS = [
        'start' => 100,
        'name' => 'group_options',
    ];

    const GROUP_ADVANCED_OPTIONS = [
        'start' => 200,
        'name' => 'group_advanced_options',
    ];

    protected array $drupalFields = [
        'node' => [
            'title',
            'uid',
            'changed',
            'created',
            'langcode',
            'status',
            'body',

            // renamed
            'created_at', // created
            'updated_at', // changed
            'author', // uid
        ],
        'user' => [
            'title',
            'uid',
            'changed',
            'created',
            'langcode',
            'status',

            // renamed
            'created_at', // created
            'updated_at', // changed
            'author', // uid
        ],
        'paragraph' => [
            'uid',
            'changed',
            'created',
            'langcode',
            'status',

            // renamed
            'created_at', // created
            'updated_at', // changed
            'author', // uid
            'authors',
        ],
    ];

    protected array $swappableNames = [
        'created' => 'created_at',
        'changed' => 'updated_at',
        'uid' => 'authors',
        'uuid' => 'uuid_host',
    ];

    /**
     * Supported languages
     *
     * @var array|null
     */
    protected ?array $languages = null;

    /**
     * Default language
     *
     * @var mixed|null
     */
    protected $defaultLanguage = null;

    /**
     * BridgeContentType constructor.
     *
     * @param  OutputInterface|null  $console
     */
    public function __construct(OutputInterface $console = null)
    {
        parent::__construct();

        $this->output = $console;

        $this->languages = array_intersect(list_languages(), array_keys(\Drupal::languageManager()->getLanguages()));
        if (empty($this->languages)) {
            throw new \RuntimeException('No language set for this instance');
        }
        // Set default language to APP_LOCALE or first known language
        $this->defaultLanguage = env('APP_LOCALE', $this->languages[0]);
        $language = \Drupal::languageManager()->getCurrentLanguage()->getId();
        if ($this->defaultLanguage != $language) {
            \Drupal::configFactory()->getEditable('system.site')->set('default_langcode',
                $this->defaultLanguage)->save();
        }
    }

    /**
     * Get the drupal field type ( used to create its storage ) from inside type
     *
     * @param  int  $type
     *
     * @return string
     * @throws Exception
     */
    public function getDrupalTypeFromInside(int $type): string
    {
        if (array_key_exists($type, $this->typeMap)) {
            return $this->typeMap[$type];
        }
        throw new Exception('['.$type.'] is not a valid inside field type');
    }

    /**
     * Get the inside type from a storage field type
     *
     * @param  string  $type
     *
     * @return int
     * @throws Exception
     */
    public function getInsideTypeFromDrupal(string $type): int
    {
        if (in_array($type, $this->typeMap)) {
            return array_search($type, $this->typeMap);
        }
        throw new Exception('['.$type.'] is not a valid or known drupal field type');
    }

    protected function write(string $message)
    {
        $this->output?->write($message);
    }

    protected function writeln(string $message)
    {
        $this->output?->writeln($message);
    }
}
