<?php

namespace Inside\Dms\Console\Commands;

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Console\Command;
use Inside\Content\Models\Contents\Documents;
use Inside\Content\Models\Contents\Folders;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Illuminate\Support\Facades\Validator;

class ImportFoldersAndFilesInDMS extends Command
{
    /**
     * @var string
     */
    protected $name = 'inside:dms:import';

    /**
     * @var string
     */
    protected $signature = 'inside:dms:import {path : The path of directory to import}
                                              {parent_uuid : The parent Folder}
                                              {author_email=technique@maecia.com : The author of the folder}
                                              {--M|media-folder : Use if you want to set Folders as Media Folder}';

    /**
     * @var string
     */
    protected $description = 'Import folders and files in DMS';

    protected BridgeContent $bridge;

    public function __construct()
    {
        parent::__construct();

        $this->bridge = new BridgeContent();
    }

    /**
     * Handle index command
     */
    public function handle(): void
    {
        $path = $this->argument('path');
        $author_email = $this->argument('author_email');
        $parent_uuid = $this->argument('parent_uuid');
        $is_media_folder = $this->option('media-folder');

        $validator = Validator::make([
            'path' => $path,
            'author_email' => $author_email,
            'parent_uuid' => $parent_uuid,
            'is_media_folder' => $is_media_folder,
        ], [
            'path' => 'required|string',
            'author_email' => 'required|email|exists:inside_users,email',
            'parent_uuid' => 'required|string|exists:inside_content_folders,uuid',
            'is_media_folder' => 'required|boolean',
        ]);

        if ($validator->fails()) {
            $this->error($validator->errors()->first());

            return;
        }

        /** @var string $path */
        $path = $validator->validated()['path'];

        /** @var string $author_email */
        $author_email = $validator->validated()['author_email'];

        /** @var string $parent_uuid */
        $parent_uuid = $validator->validated()['parent_uuid'];

        /** @var bool $is_media_folder */
        $is_media_folder = $validator->validated()['is_media_folder'];

        $user = Users::where('email', $author_email)->first();

        $folder = Folders::where('uuid', $parent_uuid)->first();

        if (str($path)->start('./')) {
            $path = cms_base_path(str($path)->replace('./', '')->toString());
        }

        if (!file_exists($path)) {
            $this->error("The directory $path do not exist");

            return;
        }

        $this->processing($path, $user, $folder, $is_media_folder);
    }

    protected function processing(string $contentPath, Users $user, Folders $folder, bool $isMediaFolder): void
    {
        foreach ($this->getFolderContent($contentPath) as $content) {
            $path = "$contentPath/$content";
            if (is_dir($path)) {
                $this->importFolder($path, $content, $user, $folder, $isMediaFolder);
            } else {
                $this->importFile($path, $content, $user, $folder);
            }
        }
    }

    protected function importFolder(string $contentPath, string $title, Users $user, Folders $folder, bool $isMediaFolder): void
    {
        $query = Folders::whereHas('folders', function ($query) use ($folder) {
            $query->where('uuid', $folder->uuid);
        })->where('title', $title);

        if ($query->exists()) {
            $folder = $query->first();

            $this->comment("\nFolder $title already exists");
        } else {
            $uuid = $this->bridge->contentInsert(
                'folders',
                [
                    'type' => 'node',
                    'bundle' => 'folders',
                    'title' => $title,
                    'folders' => $folder->uuid,
                    'author' => $user->uuid,
                    'is_media_folder' => $isMediaFolder,
                ]
            );

            $folder = Folders::where('uuid', $uuid)->first();

            $this->info("\nFolder $title imported");
        }

        $this->processing($contentPath, $user, $folder, $isMediaFolder);
    }

    protected function importFile(string $contentPath, string $title, Users $user, Folders $folder): void
    {
        $query = Documents::whereHas('folders', function ($query) use ($folder) {
            $query->where('uuid', $folder->uuid);
        })->where('title', $title);

        if ($query->exists()) {
            $this->comment(' ├─> File ' . $title . ' already exists');

            return;
        }

        /** @var string $content */
        $content = file_get_contents($contentPath);
        $chunkId = Str::random(32);
        $chunkPath = "chunks/$chunkId/$title";
        Storage::makeDirectory("chunks/$chunkId");
        Storage::disk('local')->put($chunkPath, $content);

        $this->bridge->contentInsert(
            'documents',
            [
                'type' => 'node',
                'bundle' => 'documents',
                'title' => $title,
                'author' => $user->uuid,
                'file' => $chunkPath,
                'folders' => $folder->uuid,
            ]
        );

        $this->info(' ├─> File ' . $title . ' imported');
    }

    protected function getFolderContent(string $path): array
    {
        $scan = scandir($path);

        if ($scan === false) {
            return [];
        }

        return array_diff($scan, ['.', '..', '.DS_Store']);
    }
}
