<?php

namespace Inside\Course\Services;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Inside\Course\Enums\NotificationChannel;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Content\Models\Contents\Courses;
use Inside\Authentication\Models\User;
use Inside\Notify\Models\NotificationSubscriber;
use Inside\Notify\Models\NotificationType;
use Inside\Course\Models\Course;
use Inside\Course\Notifications\CourseReminderMailNotification;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Log;
use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Support\Carbon;

class CourseReminderService extends BaseNotificationListener
{
    public function notifyUsers(): void
    {
        $roleUsers = User::query()
            ->join('inside_users_roles as ur', 'ur.user_uuid', '=', 'inside_users.uuid')
            ->join('inside_courses_roles as cr', 'cr.role_id', '=', 'ur.role_id')
            ->select(['inside_users.*']);

        $progressUsers = User::query()
            ->join('inside_courses_users_progressions as p', 'p.user_uuid', '=', 'inside_users.uuid')
            ->where('p.ratio', '>', 0)->where('p.ratio', '<', 1)
            ->select(['inside_users.*']);

        $users = User::query()
            ->fromSub($roleUsers->union($progressUsers), 'u')
            ->distinct()
            ->get();

        /**
         * @var Collection<NotificationType>
         */
        $notificationTypes = NotificationType::where([
            'action' => 'reminder',
        ])->get();

        $users->each(function ($user) use ($notificationTypes) {
            try {
                $courses = $this->getReminderCourses($user->uuid);

                if ($courses->isEmpty()) {
                    return;
                }

                foreach ($notificationTypes as $notificationType) {
                    $this->route = $user;
                    if ($notificationType->via === NotificationChannel::WEB) {
                        $this->sendWebNotification($user, $notificationType);
                    } else {
                        $notification = new CourseReminderMailNotification($notificationType, $user, null, []);
                        $this->route->notify($notification);
                    }
                }
            } catch (\Exception $exception) {
                Log::error('Reminder-Email '. $exception->getMessage());
            }
        });
    }

    public function getReminderCourses(string $userUuid): Collection
    {
        return Course::query()
            ->from('inside_courses as c')
            ->where('c.status', 1)
            ->with('model')
            ->leftJoin('inside_courses_users_progressions as p', function ($join) use ($userUuid) {
                $join->on('p.course_id', '=', 'c.id')
                    ->where('p.user_uuid', '=', $userUuid)
                    ->where('p.ratio', '>', 0)
                    ->where('p.ratio', '<', 1);
            })
            ->leftJoin('inside_courses_users_progressions as completed', function ($join) use ($userUuid) {
                $join->on('completed.course_id', '=', 'c.id')
                    ->where('completed.user_uuid', '=', $userUuid)
                    ->where('completed.ratio', '=', 1);
            })
            ->leftJoin('inside_courses_roles as cr', 'cr.course_id', '=', 'c.id')
            ->leftJoin('inside_users_roles as ur', function ($join) use ($userUuid) {
                $join->on('ur.role_id', '=', 'cr.role_id')
                    ->where('ur.user_uuid', '=', $userUuid);
            })
            ->whereNull('completed.id')
            ->where(function ($q) {
                $q->whereNotNull('p.id')
                    ->orWhereNotNull('ur.user_uuid');
            })
            ->select([
                'c.*',
                DB::raw('CASE WHEN p.id IS NOT NULL THEN 1 ELSE 0 END AS in_progress'),
                DB::raw('COALESCE(p.ratio, 0) AS current_ratio'),
            ])
            ->distinct()
            ->get();
    }

    public function scheduleReminderCommand(Schedule $schedule): void
    {
        $isEnabled = (int) setting('courses', 'reminder_email_enabled', '0');
        if ($isEnabled !== 1) {
            return;
        }

        $type = setting('courses', 'reminder_email_recurrence_type', 'weekly'); // weekly|monthly
        $day = (int) setting('courses', 'reminder_email_day', '1'); // 1..7 (Mon..Sun)
        $time = setting('courses', 'reminder_email_hour', '10:00');
        $dayOfWeekMap = [
            1 => Carbon::MONDAY, 2 => Carbon::TUESDAY, 3 => Carbon::WEDNESDAY,
            4 => Carbon::THURSDAY, 5 => Carbon::FRIDAY, 6 => Carbon::SATURDAY, 7 => Carbon::SUNDAY,
        ];
        $weekday = $dayOfWeekMap[$day] ?? Carbon::MONDAY;

        $event = $schedule->command('inside:courses:reminder:send')
            ->withoutOverlapping()
            ->onOneServer()
            ->description('Send course reminders')
            ->timezone('Europe/Paris');

        if ($type === 'weekly') {
            $event->weeklyOn($weekday, $time);
        } else {
            $event->weeklyOn($weekday, $time)->when(function () use ($weekday) {
                $now = Carbon::now();
                return $now->isSameDay($now->copy()->nthOfMonth(1, $weekday));
            });
        }
    }

    private function sendWebNotification(
        User $user,
        NotificationType $notificationType
    ): void {
        NotificationSubscriber::updateOrCreate([
            'user_uuid'            => $user->uuid,
            'notification_type_id' => $notificationType->id,
        ]);
        $this->notify($notificationType, $user, null);
    }
}
