<?php

namespace Inside\Course\Notifications;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Course\Facades\CourseReminder;
use Inside\Notify\Messages\MailMessage;
use Inside\Notify\Models\NotificationType;
use Inside\Notify\Notifications\MailNotification;
use Illuminate\Support\Facades\Lang;

class CourseReminderMailNotification extends MailNotification implements ShouldQueue
{
    public function __construct(NotificationType $notificationType, ?User $user, ?Content $model, ?array $data = [], ?string $theme = null)
    {
        parent::__construct($notificationType, $user, $model, $data, $theme);
    }

    /**
     * @param mixed $notifiable
     * @return MailMessage
     *
     * This argument is here because this function is an override
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function toMail($notifiable): MailMessage
    {
        $mail = $this->prepareMail();

        return $this->createMailMessage($mail);
    }

    public function prepareMail(): array
    {
        if (is_null($this->user)) {
            throw new Exception('MailNotification needs a user recipient');
        }

        /** @var array{view: string} $data */
        $data = $this->notificationType->data;
        $view = $data['view'];

        if (! is_null($this->user->langcode)) {
            Lang::setLocale($this->user->langcode);
        }

        $langcode = $this->user->langcode ?? config('app.locale');
        $contentType = class_to_type($this->notificationType->model);

        $courses = CourseReminder::getReminderCourses($this->user->uuid);

        if ($courses->isEmpty()) {
            return [];
        }

        $subject = $this->getTranslationFromSettings($view, 'subject', $langcode, $contentType);
        $inProgressTitle = $this->getTranslationFromSettings($view, 'in_progress_title', $langcode, $contentType);
        $notStartedTitle = $this->getTranslationFromSettings($view, 'not_started_title', $langcode, $contentType);
        $inProgressButtonTitle = $this->getTranslationFromSettings($view, 'in_progress_button_title', $langcode, $contentType);
        $notStartedButtonTitle = $this->getTranslationFromSettings($view, 'not_started_button_title', $langcode, $contentType);

        [$inProgressCourses, $notStartedCourses] = $courses->partition(function ($course) {
            return $course->in_progress == 1;
        });

        return [
            'buttonText' => 'button Texxt',
            'data' => [
                'inProgressTitle'           => $inProgressTitle,
                'inProgressButtonTitle'     => $inProgressButtonTitle,
                'notStartedButtonTitle'     => $notStartedButtonTitle,
                'notStartedTitle'           => $notStartedTitle,
                'inProgressCourses'         => $inProgressCourses->isEmpty() ? [] : $inProgressCourses,
                'notStartedCourses'         => $notStartedCourses->isEmpty() ? [] : $notStartedCourses,
            ],
            'subject' => $subject,
            'url' => '',
            'view' => $view,
        ];
    }
}
