<?php

declare(strict_types=1);

namespace Inside\Course\Models;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Database\Eloquent\Builder;
use Inside\Database\Eloquent\WithEnhancedBuilder;

/**
 * @property int $id
 * @property string $title
 * @property string $model_uuid
 * @property string $model_type
 * @property string $langcode
 * @property boolean $status
 * @property int $created_at
 * @property int $weight
 * @property int $updated_at
 * @property-read Content $model
 * @property Chapter $chapter
 * @property Collection|StepValidation[] $validations
 *
 * @method static static create(array $attributes = [])
 * @method static static updateOrCreate(array $attributes, array $values = [])
 * @method static static|null firstWhere($column, $operator = null, $value = null, $boolean = 'and')
 * @method static Builder query()
 * @method static Builder where($column, $operator = null, $value = null, $boolean = 'and')
 * @method static Builder|static ofCourse(Course $course, bool $onlyPublished = true)
 * @method static Builder|static ofChapter(Chapter $chapter, bool $onlyPublished = true)
 * @method static Builder|static onlyPublished(bool $onlyPublished = true)
 * @method static Builder|static validatedBy(User $user, bool $onlyPublished = true)
 */
class Step extends Model
{
    use WithEnhancedBuilder;

    public const CONTENT_TYPES = [
        'courses_lessons',
        'courses_quizzes',
    ];

    /**
     * @var string
     */
    protected $table = 'inside_courses_steps';

    /**
     * @var string[]
     */
    protected $fillable = [
        'model_uuid',
        'model_type',
        'title',
        'weight',
        'langcode',
        'status',
        'created_at',
        'updated_at',
        'chapter_id',
    ];

    /**
     * @var array
     */
    protected $casts = [
        'created_at' => 'timestamp',
        'updated_at' => 'timestamp',
    ];

    public function model(): MorphTo
    {
        return $this->morphTo('model', 'model_type', 'model_uuid');
    }

    public function chapter(): BelongsTo
    {
        return $this->belongsTo(Chapter::class);
    }

    public function validations(): HasMany
    {
        return $this->hasMany(StepValidation::class);
    }

    public function validate(User $user): StepValidation
    {
        return StepValidation::updateOrCreate(['user_uuid' => $user->uuid, 'step_id' => $this->id], ['validated' => true]);
    }

    public function invalidate(User $user): ?StepValidation
    {
        $validation = $this->userValidation($user);

        if (! $validation instanceof StepValidation) {
            return null;
        }

        $validation->update(['validated' => false]);

        return $validation;
    }

    public function userValidation(User $user): ?StepValidation
    {
        /** @var ?StepValidation $validation */
        $validation = $this->validations()->firstWhere('user_uuid', $user->uuid);

        return $validation;
    }

    public function scopeOnlyPublished(Builder $query, bool $onlyPublished = true): Builder
    {
        return $query->when(
            $onlyPublished,
            fn (Builder $query) => $query
                ->where('status', true)
                ->whereHas('chapter', fn (Builder $query) => $query->where('status', true))
        );
    }

    public function scopeOfCourse(Builder $query, Course $course, bool $onlyPublished = true): Builder
    {
        return $query
            ->onlyPublished($onlyPublished)
            ->whereHas('chapter.course', fn (Builder $query) => $query->where('course_id', $course->id));
    }

    public function scopeOfChapter(Builder $query, Chapter $chapter, bool $onlyPublished = true): Builder
    {
        return $query
            ->onlyPublished($onlyPublished)
            ->where('chapter_id', $chapter->id);
    }

    public function scopeValidatedBy(Builder $query, User $user, bool $onlyPublished = true): Builder
    {
        return $query
            ->onlyPublished($onlyPublished)
            ->whereHas(
                'validations',
                fn (Builder $query) => $query
                    ->where('user_uuid', $user->uuid)
                    ->where('validated', true)
            );
    }
}
