<?php

declare(strict_types=1);

namespace Inside\Course\Http\Middlewares;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\Schema;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Content;
use Inside\Course\Models\Course;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\PermissionSchema;
use Inside\Permission\Facades\Role as RoleService;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Str;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Log;

final class RolesMiddleware
{
    public function handle(Request $request, Closure $next): mixed
    {
        $method = $request->getMethod();
        $path = $request->path();

        if (
            Str::is('api/v1/content/courses/*', $path)
            && in_array($method, ['POST', 'PUT'])
        ) {
            $roles = $request->get('roles');
            $courseUuid = Str::afterLast($path, '/');
            unset($request['roles']);

            $course = Course::where('model_uuid', $courseUuid)->first();
            $course->roles()->sync($roles);
        }

        $response = $next($request);
        if ($response instanceof Response && $response->getStatusCode() != 200) {
            return $response;
        }

        $user = Auth::user();

        preg_match('/^api\/v1\/form\/(.*)?$/', $path, $match);
        if (! isset($match[1]) || ! $user) {
            return $response;
        }

        $exploded = explode('/', $match[1]);
        $uuid = null;

        if (isset($exploded[1])) {
            $match[1] = $exploded[0];
            $uuid = $exploded[1];
        }

        if ($match[1] !== 'courses' || ! InsideSchema::hasModel($match[1])) {
            return $response;
        }

        $courseRolesIds = [];
        try {
            if ($uuid) {
                $courseRoles = Course::where('model_uuid', $uuid)->firstOrFail();
                $courseRolesIds = $courseRoles->roles()->pluck('role_id')->toArray();
            }
        } catch (ModelNotFoundException $e) {
            Log::info('Roles-Middleware ' . $e->getMessage());
        }

        $rolesDescriptions = [];
        foreach (list_languages() as $locale) {
            $rolesDescriptions[$locale] = trans('notifications.forms.courses.roles.info', [], $locale);
        }

        $fields = [];
        $fields[] = Permission::getRolePickerFieldForFrontForm(
            'roles',
            'notifications.forms.title.roles.label',
            null,
            $courseRolesIds,
            51,
            true,
            $rolesDescriptions
        );

        $groupRolesLabels = [];
        foreach (list_languages() as $locale) {
            $groupRolesLabels[$locale] = __('notifications.forms.courses.roles', [], $locale);
        }

        $config = [
            'id' => 'group_roles',
            'weight' => 50,
            'type' => 'fieldset',
            'classes' => '',
            'label' => $groupRolesLabels,
            'fields' => $fields,
        ];

        $data = json_decode_response($response);
        $data['data'][] = $config;

        set_response($response, $data);

        return $response;
    }
}
