<?php

declare(strict_types=1);

namespace Inside\Course\Http\Middlewares;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Course\Models\Course;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

final class InjectCoursesProgressionMiddleware
{
    private ?array $filters = null;

    public function handle(Request $request, Closure $next): mixed
    {
        if (! $this->isCourseContentRequest($request)) {
            return $next($request);
        }

        if ($this->hasCourseProgressionFilter($request)) {
            $this->handleCourseProgressionFilter($request);
        }

        $response = $next($request);

        if ($response instanceof Response && $response->isOk()) {
            return $this->injectCoursesProgression($response);
        }

        return $response;
    }

    public function getFilters(Request $request): array
    {
        if ($this->filters !== null) {
            return $this->filters;
        }

        /** @var string $filters */
        $filters = $request->query('filters') ?? '';
        $filters = json_decode($filters, true) ?? [];

        return $this->filters ??= $filters;
    }

    public function isCourseContentRequest(Request $request): bool
    {
        $path = $request->path();
        $method = $request->getMethod();

        if ($method !== 'GET') {
            return false;
        }

        if ($path === 'api/v1/search') {
            $filters = $this->getFilters($request);

            return in_array('courses', Arr::wrap($filters['in'] ?? []));
        }

        if ($path === 'api/v1/content/courses') {
            return true;
        }

        return false;
    }

    public function hasCourseProgressionFilter(Request $request): bool
    {
        return array_key_exists('course_progression', $this->getFilters($request));
    }

    public function handleCourseProgressionFilter(Request $request): void
    {
        $user = Auth::user();

        if (! $user instanceof User) {
            return;
        }

        $filters = $this->getFilters($request);

        $whitelist = match ($filters['course_progression']) {
            'not_started' => Course::notStarted($user)->pluck('model_uuid'),
            'started' => Course::started($user)->pluck('model_uuid'),
            'finished' => Course::finished($user)->pluck('model_uuid'),
            default => throw new BadRequestHttpException("Invalid value for filter 'course_progression'"),
        };

        unset($filters['course_progression']);

        $filters['uuid:in'] = $whitelist
            ->merge(Arr::wrap($filters['uuid:in'] ?? []))
            ->unique()
            ->filter()
            ->values()
            ->all();

        $inputs = $request->all();
        $inputs['filters'] = json_encode($filters);

        $request->replace($inputs);
    }

    public function injectCoursesProgression(Response $response): Response
    {
        $user = Auth::user();

        if (! $user instanceof User) {
            return $response;
        }

        $data = json_decode_response($response);

        $data['data'] = collect($data['data'])
            ->map(function (array $item) use ($user) {
                if (! isset($item['uuid'])) {
                    return $item;
                }

                $course = Course::firstWhere('model_uuid', $item['uuid']);

                if (! $course instanceof Course) {
                    return $item;
                }

                $userProgression = $course->userProgression($user);

                $item['course_progression'] = [
                    'total_steps' => $course->total_steps,
                    'total_steps_validated' => $userProgression?->validated_steps ?? 0,
                    'progression_ratio' => $userProgression?->ratio ?? 0,
                ];

                return $item;
            })
            ->all();

        set_response($response, $data);

        return $response;
    }
}
