<?php

declare(strict_types=1);

namespace Inside\Course\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Inside\Content\Models\Contents\Courses;
use Inside\Course\Contracts\CourseProgression;
use Inside\Course\Models\Course;
use Inside\Course\Models\UserProgression;
use Inside\Host\Bridge\BridgeContent;
use Inside\User\Models\User;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

final class CoursesController extends Controller
{
    public function publish(string $uuid): JsonResponse
    {
        /** @var Courses $course */
        $course = Courses::findOrFail($uuid);

        if (! $course->published) {
            (new BridgeContent())->contentUpdate($course->content_type, ['uuid' => $course->uuid, 'published' => true]);
        }

        return response()->json(['published' => true]);
    }

    public function unpublish(string $uuid): JsonResponse
    {
        /** @var Courses $course */
        $course = Courses::findOrFail($uuid);

        if ($course->published) {
            (new BridgeContent())->contentUpdate($course->content_type, ['uuid' => $course->uuid, 'published' => false]);
        }

        return response()->json(['published' => false]);
    }

    public function visit(string $uuid, CourseProgression $courseProgression): JsonResponse
    {
        /** @var Courses $course */
        $course = Courses::findOrFail($uuid);

        if (! $course->published) {
            return response()->json(['visited' => false, 'message' => 'Only published courses can be visited']);
        }

        $course = Course::firstWhere('model_uuid', $uuid);

        if (! $course instanceof Course) {
            throw new NotFoundHttpException("Course '$uuid' not found.");
        }

        /** @var User $user */
        $user = Auth::user();

        if (! $course->userProgression($user) instanceof UserProgression) {
            $courseProgression->computeUserProgression($course, $user);
        }

        return response()->json(['visited' => true]);
    }
}
