<?php

declare(strict_types=1);

namespace Inside\Course\Services;

use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Contracts\Pagination\Paginator;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Collection;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Models\Contents\Users;
use Inside\Course\Contracts\CourseUsersStatistics as CourseUsersStatisticsContract;
use Inside\Course\Models\Course;
use Inside\Course\Models\CourseUsersStatistics;
use Inside\Database\Eloquent\Builder;
use Inside\Support\Str;

final class CourseUsersStatisticsService implements CourseUsersStatisticsContract
{
    public function getCourseUsersStatistics(?string $search = null, array $filters = []): Paginator | Collection | LengthAwarePaginator | array
    {
        $query = $this->getCourseUsersStatisticsQuery($search, $filters);

        return tap(ContentHelper::getResultFromQuery($query, $filters, true, $query->get()->count()))
            ->transform(fn (Users $user) => $this->formatUser($user));
    }

    public function getCourseUsersStatisticsQuery(?string $search = null, array $filters = []): Builder
    {
        $courseUsersStatisticTable = app(CourseUsersStatistics::class)->getTable();
        $usersModelTable = class_to_table(Users::class);

        $fullNameStatement = "CONCAT_WS(' ', $usersModelTable.firstname, $usersModelTable.lastname)";

        $query = Users::query()
            ->join($courseUsersStatisticTable, "$usersModelTable.uuid", '=', "$courseUsersStatisticTable.user_uuid")
            ->where("$courseUsersStatisticTable.viewed", true)
            ->select([
                "$usersModelTable.uuid",
                "$usersModelTable.email",
                "$usersModelTable.firstname",
                "$usersModelTable.lastname",
                "$usersModelTable.image",
            ])
            ->selectRaw("$fullNameStatement as full_name")
            ->selectRaw("COUNT(CASE WHEN $courseUsersStatisticTable.started = 1 THEN $courseUsersStatisticTable.course_id END) as started")
            ->selectRaw("COUNT(CASE WHEN $courseUsersStatisticTable.finished = 1 THEN $courseUsersStatisticTable.course_id END) as finished")
            ->when(
                $search,
                fn (Builder $query, string $search) => $query->where(
                    fn (Builder $query) => $query
                        ->whereRawLike($fullNameStatement, Str::wrap($search, '%'))
                        ->orWhereLike("$usersModelTable.email", Str::wrap($search, '%'))
                )
            )
            ->groupBy("$usersModelTable.uuid");

        ContentHelper::applySortToQuery($query, $filters, 'full_name', 'asc', [
            'email',
            'started',
            'finished',
        ]);

        return $query;
    }

    public function getCourseUsersDetailStatisticsQuery(string $userUuid, ?string $search = null, array $filters = []): Builder
    {
        $courseUsersStatisticsTable = app(CourseUsersStatistics::class)->getTable();
        $courseTable = app(Course::class)->getTable();
        $usersModelTable = app(Users::class)->getTable();

        $query = Course::query()
            ->crossJoin($usersModelTable)
            ->leftJoin(
                $courseUsersStatisticsTable,
                fn (JoinClause $join) => $join
                    ->on("$courseUsersStatisticsTable.course_id", '=', "$courseTable.id")
                    ->on("$courseUsersStatisticsTable.user_uuid", '=', "$usersModelTable.uuid")
            )
            ->select([
                "$courseTable.model_uuid",
                "$courseTable.title",
                "$courseUsersStatisticsTable.finished",
                "$courseUsersStatisticsTable.started",
                "$courseUsersStatisticsTable.viewed",
                "$courseUsersStatisticsTable.ratio",
            ])
            ->where("$usersModelTable.uuid", $userUuid)
            ->when($search, fn (Builder $query, string $search) => $query->whereLike("$courseTable.title", Str::wrap($search, '%')));

        collect($filters)
            ->each(fn (mixed $value, string $filter) => match ($filter) {
                'finished', 'started' => $query->where($filter, $value),
                'viewed' => $query->when(
                    $value,
                    fn ($query) => $query->where('viewed', true),
                    fn ($query) => $query->where(
                        fn ($query) => $query
                            ->where('viewed', false)
                            ->orWhereNull('viewed')
                    )
                ),
                default => null,
            });

        $query->withCasts([
            'viewed' => 'boolean',
            'started' => 'boolean',
            'finished' => 'boolean',
        ]);

        ContentHelper::applySortToQuery($query, $filters, 'title', 'asc', [
            'viewed',
            'started',
            'finished',
        ]);

        return $query;
    }

    public function getCourseUsersDetailStatistics(string $userUuid, ?string $search = null, array $filters = []): Paginator | Collection | LengthAwarePaginator | array
    {
        $query = $this->getCourseUsersDetailStatisticsQuery($userUuid, $search, $filters);

        return ContentHelper::getResultFromQuery($query, $filters, true, $query->count());
    }

    private function formatUser(Users $user): array
    {
        $image = protected_file_url(content: $user, fieldName: 'image');

        return [
            'uuid' => $user->uuid,
            'full_name' => trim(Str::title($user->firstname).' '.Str::upper($user->lastname)) ?: null,
            'email' => $user->email,
            'image' => $image ? url($image) : null,
            'started' => $user->started,
            'finished' => $user->finished,
            'viewed' => $user->viewed,
        ];
    }
}
