<?php

declare(strict_types=1);

namespace Inside\Course\Services;

use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Contracts\Pagination\Paginator;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Collection;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Models\Contents\Courses;
use Inside\Content\Models\Contents\Users;
use Inside\Course\Contracts\CourseStatistics as CourseStatisticsContract;
use Inside\Course\Models\Course;
use Inside\Course\Models\CourseStatistics;
use Inside\Course\Models\CourseUsersStatistics;
use Inside\Database\Eloquent\Builder;
use Inside\Support\Str;

final class CourseStatisticsService implements CourseStatisticsContract
{
    public function getCoursesStatistics(?string $search = null, array $filters = []): Paginator | Collection | LengthAwarePaginator | array
    {
        $query = $this->getCoursesStatisticsQuery($search, $filters);

        return ContentHelper::getResultFromQuery($query, $filters, true, $query->count());
    }

    public function getCoursesStatisticsQuery(?string $search = null, array $filters = []): Builder
    {
        $courseTable = app(Course::class)->getTable();
        $courseStatisticsTable = app(CourseStatistics::class)->getTable();
        $coursesModelTable = app(Courses::class)->getTable();

        $query = CourseStatistics::query()
            ->join($courseTable, "$courseTable.id", '=', "$courseStatisticsTable.course_id")
            ->join($coursesModelTable, 'uuid', '=', 'model_uuid')
            ->select([
                "$courseStatisticsTable.id",
                "$courseStatisticsTable.course_id",
                "$courseStatisticsTable.viewed",
                "$courseStatisticsTable.not_viewed",
                "$courseStatisticsTable.finished",
                "$courseStatisticsTable.not_finished",
                "$courseStatisticsTable.average_ratio",
                "$coursesModelTable.published",
                "$coursesModelTable.status",
                "$coursesModelTable.title",
                "$coursesModelTable.created_at",
                "$courseTable.model_uuid",
            ])
            ->withCasts([
                'published' => 'boolean',
                'status' => 'status',
                'created_at' => 'timestamp',
            ])
            ->when($search, fn (Builder $query, string $search) => $query->whereLike("$coursesModelTable.title", Str::wrap($search, '%')));

        collect($filters)
            ->each(fn (mixed $value, string $filter) => match ($filter) {
                'courses_categories' => $query->when(
                    $value,
                    fn (Builder $query, mixed $category) => $query->whereHas(
                        'course.model.coursesCategories',
                        fn (Builder $query) => $query->where('uuid', $category)
                    )
                ),
                'viewed', 'not_viewed', 'finished', 'not_finished' => $query->when(
                    $value,
                    fn (Builder $query, mixed $value) => $query->where($filter, $value)
                ),
                default => null,
            });

        ContentHelper::applySortToQuery($query, $filters, 'published_at', 'desc', [
            'title',
            'viewed',
            'not_viewed',
            'finished',
            'not_finished',
            'average_ratio',
            'published',
            'status',
            'created_at',
        ]);

        return $query;
    }

    public function getCoursesDetailStatisticsQuery(string $courseUuid, ?string $search = null, array $filters = []): Builder
    {
        $courseUsersStatisticsTable = app(CourseUsersStatistics::class)->getTable();
        $courseTable = app(Course::class)->getTable();
        $usersModelTable = class_to_table(Users::class);

        $fullNameStatement = "CONCAT_WS(' ', $usersModelTable.firstname, $usersModelTable.lastname)";

        $viewed = $filters['viewed'] ?? true;

        $query = Users::query()
            ->crossJoin($courseTable)
            ->leftJoin(
                $courseUsersStatisticsTable,
                fn (JoinClause $join) => $join
                    ->on("$courseTable.id", '=', "$courseUsersStatisticsTable.course_id")
                    ->on("$usersModelTable.uuid", '=', "$courseUsersStatisticsTable.user_uuid")
            )
            ->withCasts([
                'viewed' => 'boolean',
                'finished' => 'boolean',
                'started' => 'boolean',
            ])
            ->where("$usersModelTable.status", true)
            ->where("$usersModelTable.is_maintenance", false)
            ->where("$courseTable.model_uuid", $courseUuid)
            ->when(
                $viewed,
                fn ($query) => $query->where('viewed', true),
                fn ($query) => $query->where(
                    fn ($query) => $query
                        ->where('viewed', false)
                        ->orWhereNull('viewed')
                ),
            )
            ->when(
                $search,
                fn (Builder $query, string $search) => $query->where(
                    fn (Builder $query) => $query
                        ->whereRawLike($fullNameStatement, Str::wrap($search, '%'))
                        ->whereLike("$usersModelTable.email", Str::wrap($search, '%'))
                )
            );

        collect($filters)
            ->each(fn (mixed $value, string $filter) => match ($filter) {
                'finished', 'started' => $query->where($filter, $value),
                default => null,
            });

        ContentHelper::applySortToQuery($query, $filters, 'full_name', 'asc', [
            'title',
            'viewed',
            'started',
            'finished',
            'ratio',
        ]);

        return $query;
    }

    public function getCoursesDetailStatistics(string $courseUuid, ?string $search = null, array $filters = []): Paginator | Collection | LengthAwarePaginator | array
    {
        $query = $this->getCoursesDetailStatisticsQuery($courseUuid, $search, $filters);

        return tap(ContentHelper::getResultFromQuery($query, $filters, true, $query->count()))
            ->transform(fn (Users $user) => $this->formatUser($user));
    }

    private function formatUser(Users $user): array
    {
        $image = protected_file_url(content: $user, fieldName: 'image');

        return [
            'uuid' => $user->uuid,
            'full_name' => trim(Str::title($user->firstname).' '.Str::upper($user->lastname)) ?: null,
            'email' => $user->email,
            'image' => $image ? url($image) : null,
            'viewed' => $user->viewed,
            'started' => $user->started,
            'finished' => $user->finished,
            'ratio' => $user->ratio,
        ];
    }
}
