<?php

declare(strict_types=1);

namespace Inside\Course\Services;

use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Users;
use Inside\Course\Contracts\CourseStatisticsComputer;
use Inside\Course\Models\Course;
use Inside\Course\Models\CourseStatistics;
use Inside\Course\Models\CourseUsersStatistics;

final class CourseStatisticsComputerService implements CourseStatisticsComputer
{
    private const RATIO_ROUND = 2;

    public function computeCourseStatistics(Course $course): CourseStatistics
    {
        $users = CourseUsersStatistics::ofCourse($course)->pluck('user_uuid');

        $notViewed = Users::query()
            ->whereNotIn('uuid', $users)
            ->where('status', true)
            ->where('is_maintenance', '!=', true)
            ->count();

        $notViewed += CourseUsersStatistics::ofCourse($course)->where('viewed', false)->count();

        $viewed = CourseUsersStatistics::ofCourse($course)->where('viewed', true)->count();
        $notFinished = CourseUsersStatistics::ofCourse($course)->where('started', true)->where('finished', false)->count();
        $finished = CourseUsersStatistics::ofCourse($course)->where('finished', true)->count();
        $averageRatio = CourseUsersStatistics::ofCourse($course)->avg('ratio');

        return CourseStatistics::updateOrCreate(
            ['course_id' => $course->id],
            [
                'viewed' => $viewed,
                'not_viewed' => $notViewed,
                'finished' => $finished,
                'not_finished' => $notFinished,
                'average_ratio' => round((float) $averageRatio, self::RATIO_ROUND),
            ]
        );
    }

    public function computeCourseUsersStatistics(Course $course, User $user): CourseUsersStatistics
    {
        $progression = $course->userProgression($user);

        return CourseUsersStatistics::updateOrCreate(
            ['course_id' => $course->id, 'user_uuid' => $user->uuid],
            [
                'viewed' => $progression !== null,
                'started' => $progression?->ratio > 0.0,
                'finished' => $progression?->ratio === 1.0,
                'ratio' => (float) $progression?->ratio,
            ]
        );
    }

    public function computeCompleteCourseStatistics(Course $course): void
    {
        CourseUsersStatistics::where('course_id', $course->id)->delete();

        $course
            ->progressionsWithoutMaintenance
            ->pluck('user')
            ->each(fn (User $user) => $this->computeCourseUsersStatistics($course, $user));

        $this->computeCourseStatistics($course);
    }
}
