<?php

declare(strict_types=1);

namespace Inside\Course\Services;

use Inside\Authentication\Models\User;
use Inside\Course\Contracts\CourseProgression;
use Inside\Course\Models\Chapter;
use Inside\Course\Models\Course;
use Inside\Course\Models\Step;
use Inside\Course\Models\StepValidation;
use Inside\Course\Models\UserProgression;

final class CourseProgressionService implements CourseProgression
{
    private const RATIO_ROUND = 2;

    public function getUserCourseProgression(User $user, Course $course): array
    {
        $userProgression = $course->userProgression($user);

        return [
            'title' => $course->title,
            'total_steps' => $course->total_steps,
            'total_steps_validated' => $userProgression?->validated_steps ?? 0,
            'progression_ratio' => $userProgression?->ratio ?? 0,
            'chapters' => $course->publishedChapters
                ->map(function (Chapter $chapter) use ($user) {
                    return [
                        'uuid' => $chapter->model_uuid,
                        'title' => $chapter->title,
                        'weight' => $chapter->weight,
                        'validated' => $chapter->validatedBy($user),
                        'steps' => $chapter->publishedSteps
                            ->map(function (Step $step) use ($user) {
                                $validation = $step->userValidation($user);

                                return [
                                    'uuid' => $step->model_uuid,
                                    'content_type' => class_to_type($step->model_type),
                                    'title' => $step->title,
                                    'weight' => $step->weight,
                                    'validated' => $validation?->validated ?? false,
                                    'validated_at' => $validation?->updated_at,
                                ];
                            })
                            ->all(),
                    ];
                })
                ->all(),
        ];
    }

    public function updateCourseTotalSteps(Course $course): bool
    {
        return $course->update(['total_steps' => Step::ofCourse($course)->count()]);
    }

    public function computeUserProgression(Course $course, User $user): UserProgression
    {
        $validatedSteps = Step::ofCourse($course)->validatedBy($user)->count();
        $ratio = $course->total_steps === 0 ? 0 : $validatedSteps / $course->total_steps;

        return UserProgression::updateOrCreate(
            ['course_id' => $course->id, 'user_uuid' => $user->uuid],
            ['validated_steps' => $validatedSteps, 'ratio' => round($ratio, self::RATIO_ROUND)]
        );
    }

    public function computeAllProgressions(Course $course): void
    {
        StepValidation::ofCourse($course)
            ->get()
            ->pluck('user')
            ->unique()
            ->each(fn (User $user) => $this->computeUserProgression($course, $user));
    }
}
