<?php

declare(strict_types=1);

namespace Inside\Course\Services;

use Illuminate\Support\Collection;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Courses;
use Inside\Content\Models\Contents\CoursesCategories;
use Inside\Course\Contracts\CourseNotifications;
use Inside\Database\Eloquent\Builder;
use Inside\Notify\Events\CustomNotificationEvent;
use Inside\Notify\Models\NotificationType;

final class CourseNotificationsService implements CourseNotifications
{
    public function createNotificationTypes(CoursesCategories $coursesCategory): Collection
    {
        $notifications = collect([
            'web' => [
                'title' => 'notifications.courses.published.web.title',
                'description' => 'notifications.courses.published.web.description',
                'icon' => 'news',
                'text' => 'notifications.courses.published.web.text',
                'fields' => [
                    'title',
                ],
            ],
            'email' => [
                'title' => 'notifications.courses.published.mail.title',
                'description' => 'notifications.courses.published.mail.description',
                'mail' => [
                    'subject' => 'notifications.courses.published.mail.subject',
                    'text' => 'notifications.courses.published.mail.content',
                    'buttonText' => 'notifications.courses.published.mail.buttonText',
                ],
                'fields' => [
                    'title',
                ],
            ],
        ]);

        return $notifications->map(fn (array $data, string $via) => NotificationType::firstOrCreate(
            [
                'via' => $via,
                'model' => Courses::class,
                'event' => CustomNotificationEvent::class,
                'action' => 'published',
                'type' => 'system',
                'condition' => "courses_categories:$coursesCategory->uuid",
            ],
            [
                'multiple' => false,
                'language' => true,
                'profile' => false,
                'default' => false,
                'data' => $data,
            ]
        ));
    }

    public function isSubscribed(CoursesCategories $coursesCategory, User $user): bool
    {
        return NotificationType::query()
            ->where('action', 'published')
            ->where('model', Courses::class)
            ->where('condition', "courses_categories:$coursesCategory->uuid")
            ->whereHas('subscribers', fn (Builder $query) => $query->where('user_uuid', $user->uuid))
            ->exists();
    }

    public function subscribe(CoursesCategories $coursesCategory, User $user): void
    {
        $this->createNotificationTypes($coursesCategory)->each(fn (NotificationType $notificationType) => $notificationType->subscribers()->syncWithoutDetaching($user));
    }

    public function unsubscribe(CoursesCategories $coursesCategory, User $user): void
    {
        NotificationType::query()
            ->where('action', 'published')
            ->where('model', Courses::class)
            ->where('condition', "courses_categories:$coursesCategory->uuid")
            ->each(fn (NotificationType $notificationType) => $notificationType->subscribers()->detach($user));
    }

    public function deleteNotificationTypes(CoursesCategories $coursesCategory): void
    {
        NotificationType::query()
            ->where('action', 'published')
            ->where('model', Courses::class)
            ->where('condition', "courses_categories:$coursesCategory->uuid")
            ->delete();
    }

    public function sendNotifications(Courses $course): void
    {
        $course->coursesCategories->each(fn (CoursesCategories $category) => CustomNotificationEvent::dispatch([
            'model' => $course->content_type,
            'uuid' => $course->uuid,
            'action' => 'published',
            'condition' => "courses_categories:$category->uuid",
        ]));
    }
}
