<?php

declare(strict_types=1);

namespace Inside\Course\Providers;

use Inside\Content\Events\ContentUpdatedEvent;
use Inside\Authentication\Events\FirstConnection;
use Inside\Course\Console\ComputeCourseStatisticsCommand;
use Inside\Course\Console\RecomputeCoursesProgressionCommand;
use Inside\Course\Console\SynchronizeCoursesCommand;
use Inside\Course\Console\SendReminderEmailsCommand;
use Inside\Course\Contracts\CourseNotifications;
use Inside\Course\Contracts\CourseProgression;
use Inside\Course\Contracts\CourseStatistics;
use Inside\Course\Contracts\CourseStatisticsComputer;
use Inside\Course\Contracts\CourseUsersStatistics;
use Inside\Course\Http\Middlewares\HideUnpublishedCoursesFromSearch;
use Inside\Course\Http\Middlewares\InjectCoursesProgressionMiddleware;
use Inside\Course\Http\Middlewares\InjectStepsValidationMiddleware;
use Inside\Course\Http\Middlewares\RolesMiddleware;
use Inside\Course\Listeners\CoursesPublishedListener;
use Inside\Course\Listeners\SynchronizeChapter;
use Inside\Course\Listeners\SynchronizeCourse;
use Inside\Course\Listeners\HandleFirstConnection;
use Inside\Course\Listeners\SynchronizeStep;
use Inside\Course\Models\Chapter;
use Inside\Course\Models\Course;
use Inside\Course\Models\Step;
use Inside\Course\Models\StepValidation;
use Inside\Course\Observers\ChapterObserver;
use Inside\Course\Observers\CourseObserver;
use Inside\Course\Observers\StepObserver;
use Inside\Course\Observers\StepValidationObserver;
use Inside\Course\Services\CourseNotificationsService;
use Inside\Course\Services\CourseProgressionService;
use Inside\Course\Services\CourseStatisticsComputerService;
use Inside\Course\Services\CourseStatisticsService;
use Inside\Course\Services\CourseUsersStatisticsService;
use Inside\Support\EventServiceProvider as BaseServiceProvider;
use Inside\Settings\Events\SettingSavedEvent;
use Inside\Course\Listeners\CoursesSettingsSavedListener;

final class CourseServiceProvider extends BaseServiceProvider
{
    protected array $subscribe = [
        SynchronizeCourse::class,
        SynchronizeChapter::class,
        SynchronizeStep::class,
    ];

    protected array $listen = [
        ContentUpdatedEvent::class => [
            CoursesPublishedListener::class,
        ],
        FirstConnection::class => [
            HandleFirstConnection::class,
        ],
        SettingSavedEvent::class => [
            CoursesSettingsSavedListener::class,
        ],
    ];

    protected array $commands = [
        SynchronizeCoursesCommand::class,
        RecomputeCoursesProgressionCommand::class,
        ComputeCourseStatisticsCommand::class,
        SendReminderEmailsCommand::class,
    ];

    public array $bindings = [
        CourseProgression::class => CourseProgressionService::class,
        CourseNotifications::class => CourseNotificationsService::class,
        CourseStatisticsComputer::class => CourseStatisticsComputerService::class,
        CourseStatistics::class => CourseStatisticsService::class,
        CourseUsersStatistics::class => CourseUsersStatisticsService::class,
    ];

    public function boot(): void
    {
        parent::boot();

        Course::observe(CourseObserver::class);
        Chapter::observe(ChapterObserver::class);
        Step::observe(StepObserver::class);
        StepValidation::observe(StepValidationObserver::class);
    }

    public function register(): void
    {
        $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/scheduler.php', 'scheduler');

        $this->app->middleware(
            [
                InjectStepsValidationMiddleware::class,
                InjectCoursesProgressionMiddleware::class,
                HideUnpublishedCoursesFromSearch::class,
                RolesMiddleware::class,
            ]
        );
    }
}
