<?php

declare(strict_types=1);

namespace Inside\Course\Models;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Courses;
use Inside\Database\Eloquent\Builder;
use Inside\Database\Eloquent\WithEnhancedBuilder;
use Inside\Permission\Scopes\AllowedScope;
use Inside\Permission\Models\Role;

/**
 * @property int $id
 * @property string $title
 * @property string $model_uuid
 * @property string $langcode
 * @property int $total_steps
 * @property boolean $status
 * @property int $created_at
 * @property int $updated_at
 * @property Collection|Chapter[] $chapters
 * @property Collection|Chapter[] $publishedChapters
 * @property Collection|UserProgression[] $progressions
 * @property Collection|UserProgression[] $progressionsWithoutMaintenance
 * @property-read Courses $model
 *
 * @method static static create(array $attributes = [])
 * @method static static firstOrFail(array $attributes = [])
 * @method static static updateOrCreate(array $attributes, array $values = [])
 * @method static static|null firstWhere($column, $operator = null, $value = null, $boolean = 'and')
 * @method static Builder query()
 * @method static Builder where($column, $operator = null, $value = null, $boolean = 'and')
 * @method static Builder|static notStarted(User $user)
 * @method static Builder|static started(User $user)
 * @method static Builder|static finished(User $user)
 *
 * @mixin Collection
 */
class Course extends Model
{
    use WithEnhancedBuilder;

    /**
     * @var string
     */
    protected $table = 'inside_courses';

    /**
     * @var string[]
     */
    protected $fillable = [
        'title',
        'status',
        'model_uuid',
        'total_steps',
        'langcode',
        'created_at',
        'updated_at',
    ];

    /**
     * @var array
     */
    protected $casts = [
        'created_at' => 'timestamp',
        'updated_at' => 'timestamp',
        'status' => 'boolean',
    ];

    public function model(): BelongsTo
    {
        return $this->belongsTo(Courses::class)->withoutGlobalScope(AllowedScope::class);
    }

    public function chapters(): HasMany
    {
        return $this->hasMany(Chapter::class)->orderBy('weight')->orderBy('title');
    }

    public function publishedChapters(): HasMany
    {
        return $this->chapters()->where('status', true);
    }

    public function progressions(): HasMany
    {
        return $this->hasMany(UserProgression::class);
    }

    public function progressionsWithoutMaintenance(): HasMany
    {
        return $this->progressions()->withoutMaintenance();
    }

    public function userProgression(User $user): ?UserProgression
    {
        /** @var ?UserProgression $userProgression */
        $userProgression = $this->progressions()->firstWhere('user_uuid', $user->uuid);

        return $userProgression;
    }

    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(Role::class, 'inside_courses_roles', 'course_id', 'role_id', 'id', 'id')->withTimestamps();
    }

    public function scopeNotStarted(Builder $query, User $user): Builder
    {
        return $query
            ->whereDoesntHave(
                'progressions',
                fn (Builder $query) => $query
                    ->where('user_uuid', $user->uuid)
                    ->where('ratio', '>', 0)
            );
    }

    public function scopeStarted(Builder $query, User $user): Builder
    {
        return $query
            ->whereHas(
                'progressions',
                fn (Builder $query) => $query
                    ->where('user_uuid', $user->uuid)
                    ->where('ratio', '>', 0)
                    ->where('ratio', '<', 1)
            );
    }

    public function scopeFinished(Builder $query, User $user): Builder
    {
        return $query
            ->whereHas(
                'progressions',
                fn (Builder $query) => $query
                    ->where('user_uuid', $user->uuid)
                    ->where('ratio', 1)
            );
    }
}
