<?php

declare(strict_types=1);

namespace Inside\Course\Models;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\CoursesChapters;
use Inside\Database\Eloquent\Builder;
use Inside\Database\Eloquent\WithEnhancedBuilder;
use Inside\Permission\Scopes\AllowedScope;

/**
 * @property int $id
 * @property string $title
 * @property string $model_uuid
 * @property string $langcode
 * @property boolean $status
 * @property int|null $weight
 * @property int $created_at
 * @property int $updated_at
 * @property-read CoursesChapters $model
 * @property Course $course
 * @property Collection|Step[] $steps
 * @property Collection|Step[] $publishedSteps
 *
 * @method static static create(array $attributes = [])
 * @method static static updateOrCreate(array $attributes, array $values = [])
 * @method static static|null firstWhere($column, $operator = null, $value = null, $boolean = 'and')
 * @method static Builder query()
 * @method static Builder where($column, $operator = null, $value = null, $boolean = 'and')
 *
 * @mixin Collection
 */
class Chapter extends Model
{
    use WithEnhancedBuilder;

    /**
     * @var string
     */
    protected $table = 'inside_courses_chapters';

    /**
     * @var string[]
     */
    protected $fillable = [
        'model_uuid',
        'title',
        'status',
        'langcode',
        'weight',
        'created_at',
        'updated_at',
        'course_id',
    ];

    /**
     * @var array
     */
    protected $casts = [
        'created_at' => 'timestamp',
        'updated_at' => 'timestamp',
        'status' => 'boolean',
    ];

    public function model(): BelongsTo
    {
        return $this->belongsTo(CoursesChapters::class)->withoutGlobalScope(AllowedScope::class);
    }

    public function course(): BelongsTo
    {
        return $this->belongsTo(Course::class);
    }

    public function steps(): HasMany
    {
        return $this->hasMany(Step::class)->orderBy('weight')->orderBy('title');
    }

    public function publishedSteps(): HasMany
    {
        return $this->steps()->where('status', true);
    }

    public function validatedBy(User $user): bool
    {
        return Step::ofChapter($this)->validatedBy($user)->count() === $this->publishedSteps->count();
    }
}
