<?php

declare(strict_types=1);

namespace Inside\Course\Listeners;

use Inside\Archive\Events\ContentArchivedEvent;
use Inside\Archive\Events\ContentRestoredEvent;
use Inside\Content\Events\ContentDeletedEvent;
use Inside\Content\Events\ContentFullyInsertedEvent;
use Inside\Content\Events\ContentFullyUpdatedEvent;
use Inside\Course\Models\Chapter;
use Inside\Course\Models\Step;
use Inside\Facades\Package;

final class SynchronizeStep
{
    public function handleCourseUpdateOrCreate(ContentFullyInsertedEvent | ContentFullyUpdatedEvent $event): void
    {
        if (! in_array($event->model->content_type, Step::CONTENT_TYPES)) {
            return;
        }

        $stepContent = $event->model;

        $chapter = Chapter::firstWhere('model_uuid', $stepContent->coursesChapters->first()?->uuid);

        if (! $chapter instanceof Chapter) {
            return;
        }

        Step::updateOrCreate(
            [
                'model_uuid' => $stepContent->uuid,
                'model_type' => get_class($stepContent),
            ],
            [
                'title' => $stepContent->title,
                'status' => $stepContent->status,
                'weight' => $stepContent->weight,
                'langcode' => $stepContent->langcode,
                'chapter_id' => $chapter->id,
            ]
        );
    }

    public function handleCourseDeletion(ContentDeletedEvent $event): void
    {
        if (! in_array($event->model->content_type, Step::CONTENT_TYPES)) {
            return;
        }

        Step::firstWhere([
            'model_uuid' => $event->model->uuid,
            'model_type' => get_class($event->model),
        ])?->delete();
    }

    public function handleArchivedStep(ContentArchivedEvent $event): void
    {
        /** @var \Inside\Content\Models\Content $content */
        $content = $event->content;

        if (! in_array($content->content_type, Step::CONTENT_TYPES)) {
            return;
        }

        Step::firstWhere([
            'model_uuid' => $content->uuid,
            'model_type' => get_class($content),
        ])?->update(['status' => 0]);
    }

    public function handleRestoredStep(ContentRestoredEvent $event): void
    {
        /** @var \Inside\Content\Models\Content $content */
        $content = $event->content;

        if (! in_array($content->content_type, Step::CONTENT_TYPES)) {
            return;
        }

        Step::firstWhere([
            'model_uuid' => $content->uuid,
            'model_type' => get_class($content),
        ])?->update(['status' => 1]);
    }

    public function subscribe(): array
    {
        $bindings = [
            ContentFullyInsertedEvent::class => 'handleCourseUpdateOrCreate',
            ContentFullyUpdatedEvent::class => 'handleCourseUpdateOrCreate',
            ContentDeletedEvent::class => 'handleCourseDeletion',
        ];

        if (Package::has('inside-archive')) {
            $bindings += [
                ContentArchivedEvent::class => 'handleArchivedStep',
                ContentRestoredEvent::class => 'handleRestoredStep',
            ];
        }

        return $bindings;
    }
}
