<?php

declare(strict_types=1);

namespace Inside\Course\Listeners;

use Inside\Archive\Events\ContentArchivedEvent;
use Inside\Archive\Events\ContentRestoredEvent;
use Inside\Content\Events\ContentDeletedEvent;
use Inside\Content\Events\ContentFullyInsertedEvent;
use Inside\Content\Events\ContentFullyUpdatedEvent;
use Inside\Content\Models\Contents\CoursesChapters;
use Inside\Course\Models\Chapter;
use Inside\Course\Models\Course;
use Inside\Facades\Package;

final class SynchronizeChapter
{
    public function handleCourseUpdateOrCreate(ContentFullyInsertedEvent | ContentFullyUpdatedEvent $event): void
    {
        if (! $event->model instanceof CoursesChapters) {
            return;
        }

        $chapterContent = $event->model;

        $course = Course::firstWhere('model_uuid', $chapterContent->courses->first()?->uuid);

        if (! $course instanceof Course) {
            return;
        }

        Chapter::updateOrCreate(
            ['model_uuid' => $chapterContent->uuid],
            [
                'title' => $chapterContent->title,
                'status' => $chapterContent->status,
                'weight' => $chapterContent->weight,
                'langcode' => $chapterContent->langcode,
                'course_id' => $course->id,
            ]
        );
    }

    public function handleCourseDeletion(ContentDeletedEvent $event): void
    {
        if (! $event->model instanceof CoursesChapters) {
            return;
        }

        Chapter::firstWhere('model_uuid', $event->model->uuid)?->delete();
    }

    public function handleArchivedChapter(ContentArchivedEvent $event): void
    {
        /** @var \Inside\Content\Models\Content $content */
        $content = $event->content;


        if (!$content instanceof CoursesChapters) {
            return;
        }

        Chapter::firstWhere('model_uuid', $content->uuid)?->update(['status' => 0]);
    }

    public function handleRestoredChapter(ContentRestoredEvent $event): void
    {
        /** @var \Inside\Content\Models\Content $content */
        $content = $event->content;

        if (!$content instanceof CoursesChapters) {
            return;
        }

        Chapter::firstWhere('model_uuid', $content->uuid)?->update(['status' => 1]);
    }


    public function subscribe(): array
    {
        $bindings = [
            ContentFullyInsertedEvent::class => 'handleCourseUpdateOrCreate',
            ContentFullyUpdatedEvent::class => 'handleCourseUpdateOrCreate',
            ContentDeletedEvent::class => 'handleCourseDeletion',
        ];

        if (Package::has('inside-archive')) {
            $bindings += [
                ContentArchivedEvent::class => 'handleArchivedChapter',
                ContentRestoredEvent::class => 'handleRestoredChapter',
            ];
        }

        return $bindings;
    }
}
