<?php

declare(strict_types=1);

namespace Inside\Course\Http\Middlewares;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Course\Models\Step;
use Symfony\Component\HttpFoundation\Response;

final class InjectStepsValidationMiddleware
{
    public function handle(Request $request, Closure $next): mixed
    {
        $response = $next($request);

        if ($response instanceof Response && $response->isOk() && $this->isStepContentRequest($request)) {
            return $this->injectStepsValidation($response);
        }

        return $response;
    }

    public function isStepContentRequest(Request $request): bool
    {
        $path = $request->path();
        $method = $request->getMethod();

        $stepsContentTypes = collect(Step::CONTENT_TYPES)->join('|');

        return $method === 'GET' && preg_match("#^api/v1/content/($stepsContentTypes)$#", $path);
    }

    public function injectStepsValidation(Response $response): Response
    {
        $user = Auth::user();

        if (! $user instanceof User) {
            return $response;
        }

        $data = json_decode_response($response);

        $data['data'] = collect($data['data'])
            ->map(function (array $item) use ($user) {
                if (! isset($item['uuid'])) {
                    return $item;
                }

                $validation = Step::firstWhere('model_uuid', $item['uuid'])?->userValidation($user);

                $item['course_validation'] = [
                    'validated' => $validation?->validated,
                    'validated_at' => $validation?->updated_at,
                ];

                return $item;
            })
            ->all();

        set_response($response, $data);

        return $response;
    }
}
