<?php

declare(strict_types=1);

namespace Inside\Course\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\ContentHelper;
use Inside\Course\Contracts\CourseStatistics;
use Inside\Course\Exports\CourseDetailStatisticsExport;
use Inside\Course\Exports\CoursesStatisticsExport;
use Inside\Statistics\Listeners\NotifyUserOfCompletedStatisticExport;

final class CoursesStatisticsController extends Controller
{
    public function index(Request $request, CourseStatistics $courseStatistics): JsonResponse
    {
        $search = $request->query->get('query') ?: null;
        $filters = ContentHelper::extractFiltersInputFromRequest($request);

        return response()->json($courseStatistics->getCoursesStatistics($search, $filters));
    }

    public function exportIndex(Request $request): JsonResponse
    {
        /** @var User $user */
        $user = Auth::user();

        $search = $request->query->get('query') ?: null;
        $filters = ContentHelper::extractFiltersInputFromRequest($request);

        $excelPath = 'statistics/courses_'.now('Europe/Paris')->format('Y_m_d_His').'_export.xls';

        (new CoursesStatisticsExport($search, $filters, $user->langcode))
            ->queue($excelPath, 'protected')
            ->chain([new NotifyUserOfCompletedStatisticExport($user, $excelPath, 'protected')])
            ->allOnQueue(get_low_priority_queue_name());

        return response()->json(['success' => true]);
    }


    public function detail(string $uuid, Request $request, CourseStatistics $courseStatistics): JsonResponse
    {
        $search = $request->query->get('query') ?: null;
        $filters = ContentHelper::extractFiltersInputFromRequest($request);

        return response()->json($courseStatistics->getCoursesDetailStatistics($uuid, $search, $filters));
    }

    public function exportDetail(string $uuid, Request $request): JsonResponse
    {
        /** @var User $user */
        $user = Auth::user();

        $search = $request->query->get('query') ?: null;
        $filters = ContentHelper::extractFiltersInputFromRequest($request);

        $excelPath = "statistics/course_details_{$uuid}_".now('Europe/Paris')->format('Y_m_d_His').'_export.xls';

        (new CourseDetailStatisticsExport($uuid, $search, $filters, $user->langcode))
            ->queue($excelPath, 'protected')
            ->chain([new NotifyUserOfCompletedStatisticExport($user, $excelPath, 'protected')])
            ->allOnQueue(get_low_priority_queue_name());

        return response()->json(['success' => true]);
    }
}
