<?php

declare(strict_types=1);

namespace Inside\Course\Exports;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Lang;
use Inside\Course\Contracts\CourseStatistics;
use Inside\Database\Eloquent\Builder;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;

final class CoursesStatisticsExport extends AbstractCourseStatisticsExport
{
    private ?CourseStatistics $courseStatistics = null;

    private function getCourseStatistics(): CourseStatistics
    {
        return $this->courseStatistics ??= app(CourseStatistics::class);
    }

    public function __construct(
        private ?string $search = null,
        private array $filters = [],
        private ?string $langcode = 'en'
    ) {
    }

    public function query(): Builder
    {
        return $this->getCourseStatistics()->getCoursesStatisticsQuery($this->search, $this->filters);
    }

    public function headings(): array
    {
        if ($this->langcode) {
            Lang::setLocale($this->langcode);
        }

        return [
            Lang::get('courses.statistics.export.cols.title'),
            Lang::get('courses.statistics.export.cols.created_at'),
            Lang::get('courses.statistics.export.cols.status'),
            Lang::get('courses.statistics.export.cols.viewed'),
            Lang::get('courses.statistics.export.cols.average_ratio'),
        ];
    }

    public function map($row): array
    {
        return [
            $row->title,
            Date::dateTimeToExcel(Carbon::createFromTimestamp($row->created_at)),
            $this->mapBooleanValue((bool) $row->status, ['true' => 'courses.statistics.export.values.status.true', 'false' => 'courses.statistics.export.values.status.false']),
            $row->viewed,
            $row->average_ratio * 100 .'%',
        ];
    }

    public function columnFormats(): array
    {
        return [
            'B' => NumberFormat::FORMAT_DATE_DDMMYYYY,
        ];
    }
}
