<?php

declare(strict_types=1);

namespace Inside\Course\Console;

use Illuminate\Database\Eloquent\Builder;
use Inside\Console\Command;
use Inside\Course\Contracts\CourseStatisticsComputer;
use Inside\Course\Models\Course;

final class ComputeCourseStatisticsCommand extends Command
{
    /**
     * @var string
     */
    protected $signature = 'inside:courses:stats:compute {--c|courses= : uuids of courses delimited by comma}';

    /**
     * @var string
     */
    protected $description = 'Compute statistics for all courses';

    public function handle(CourseStatisticsComputer $courseStatisticsComputer): void
    {
        $coursesUuids = str($this->option('courses') ?? '')->explode(',')->filter();

        $courses = Course::query()
            ->when($coursesUuids->isNotEmpty(), fn (Builder $query) => $query->whereIn('model_uuid', $coursesUuids))
            ->get();

        $this->getOutput()->text("Calculate courses statistics ({$courses->count()} courses)");

        $this->getOutput()->progressStart($courses->count());

        $courses->each(function (Course $course) use ($courseStatisticsComputer) {
            $courseStatisticsComputer->computeCompleteCourseStatistics($course);
            $this->getOutput()->progressAdvance();
        });
        $this->getOutput()->progressFinish();
    }
}
