<?php

namespace App\Services;

use Exception;
use Illuminate\Contracts\Process\ProcessResult;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Process;

final class GitService
{
    private string $path = '';


    private function __construct(string $path)
    {
        $this->path = $path;
    }

    public static function on(string $path): self
    {
        return new self($path);
    }

    private static function command(string ...$args): string
    {
        return collect(['LANG=en_GB', 'git', ...$args])->join(' ');
    }

    public function init(): ProcessResult
    {
        return Process::forever()->path($this->path)->run(self::command('init'));
    }

    public function remote(string $name, string $url): ProcessResult
    {
        $result = Process::forever()->path($this->path)->run(self::command('remote', 'add', $name, $url));

        if ($result->failed()) {
            return Process::forever()->path($this->path)->run(self::command('remote', 'set-url', $name, $url));
        }

        return $result;
    }

    public function add(string ...$files): ProcessResult
    {
        return Process::forever()->path($this->path)->run(self::command('add', ...$files));
    }

    public function checkout(string $name): ProcessResult
    {
        $result = Process::forever()->path($this->path)->run(self::command('checkout', $name));

        if ($result->failed()) {
            return Process::forever()->path($this->path)->run(self::command('checkout', '-b', $name));
        }

        return $result;
    }

    public function push(string $remote, string $branch): ProcessResult
    {
        return Process::forever()->path($this->path)->run(self::command('push', '-u', $remote, $branch));
    }

    public function commit(string $message): ProcessResult
    {
        return Process::forever()->path($this->path)->run(self::command('commit', '-m', str($message)->wrap('"')));
    }

    public static function clone(string $repository, string $directory, string $branch = 'develop'): ProcessResult
    {
        return Process::forever()->run(self::command('clone', '-b', $branch, $repository, $directory));
    }

    public static function pull(string $directory): ProcessResult
    {
        return Process::forever()->path($directory)->run(self::command('pull'));
    }
}
