<?php

namespace App\Services\DockerModules;

use Illuminate\Contracts\Process\ProcessResult;
use Illuminate\Support\Facades\Process;

final class DockerRunService
{
    private bool $remove = false;

    private bool $interactive = false;

    private bool $detach = false;

    private bool $tty = false;

    private string $name = '';

    private string $image = '';

    private string $command = '';

    private array $ports = [];

    private array $volumes = [];

    private array $labels = [];

    private array $networks = [];

    private function __constructor()
    {
        //
    }

    public function remove(): self
    {
        $this->remove = true;

        return $this;
    }

    public function interactive(): self
    {
        $this->interactive = true;

        return $this;
    }

    public function image(string $image): self
    {
        $this->image = $image;

        return $this;
    }

    public function port(int $host, int $container): self
    {
        $this->ports[] = [
            'host' => $host,
            'container' => $container,
        ];

        return $this;
    }

    public function tty(): self
    {
        $this->tty = true;

        return $this;
    }

    public function volume(string $host, string $container): self
    {
        $this->volumes[] = [
            'host' => $host,
            'container' => $container,
        ];

        return $this;
    }

    public function label(string $label): self
    {
        $this->labels[] = $label;

        return $this;
    }

    public function networks(string $network): self
    {
        $this->networks[] = $network;

        return $this;
    }

    public function command(string $command): self
    {
        $this->command = $command;

        return $this;
    }

    public function detach(): self
    {
        $this->detach = true;

        return $this;
    }

    public function name(string $name): self
    {
        $this->name = $name;

        return $this;
    }

    public function execute(): ProcessResult
    {
        $command = collect(['docker', 'run']);

        if ($this->remove) {
            $command->push('--rm');
        }

        if ($this->interactive) {
            $command->push('-it');
        }

        if ($this->detach) {
            $command->push('-d');
        }

        if ($this->name) {
            $command->push('--name '.$this->name);
        }

        $ports = collect($this->ports);
        if ($ports->isNotEmpty()) {
            $ports->each(fn (array $port) => $command->push('-p '.$port['host'].':'.$port['container']));
        }

        $volumes = collect($this->volumes);
        if ($volumes->isNotEmpty()) {
            $volumes->each(fn (array $volume) => $command->push('-v '.$volume['host'].':'.$volume['container']));
        }

        $labels = collect($this->labels);
        if ($labels->isNotEmpty()) {
            $labels->each(fn (string $label) => $command->push('--label '.str($label)->wrap('\'')));
        }

        $networks = collect($this->networks);
        if ($networks->isNotEmpty()) {
            $networks->each(fn (string $network) => $command->push('--network '.$network));
        }

        if ($this->image) {
            $command->push($this->image);
        }

        if ($this->command) {
            $command->push($this->command);
        }

        return Process::forever()
            ->when($this->tty, fn ($process) => $process->tty())
            ->run($command->filter()->join(' '));
    }

    public static function boot(): self
    {
        return new self();
    }
}
