<?php

namespace App\Services\DockerModules;

use Illuminate\Contracts\Process\ProcessResult;
use Illuminate\Support\Facades\Process;
use Illuminate\Support\Traits\Conditionable;

final class DockerCompose
{
    use Conditionable;

    private string $location = '';

    private string $container = '';

    private bool $detach = false;

    private bool $remove = false;

    private bool $interactive = false;

    private bool $follow = false;

    private false|int $tail = false;

    private bool $tty = false;

    private function __construct()
    {
        //
    }

    public function location(string $location): self
    {
        $this->location = $location;

        return $this;
    }

    public function detach(): self
    {
        $this->detach = true;

        return $this;
    }

    public function remove(): self
    {
        $this->remove = true;

        return $this;
    }

    public function follow(): self
    {
        $this->follow = true;

        return $this;
    }

    public function tail(int $lines): self
    {
        $this->tail = $lines;

        return $this;
    }

    public function interactive(): self
    {
        $this->interactive = true;

        return $this;
    }

    public function tty(): self
    {
        $this->tty = true;

        return $this;
    }

    public function pull(): ProcessResult
    {
        return $this->execute('pull');
    }

    public function up(): ProcessResult
    {
        return $this->execute('up');
    }

    public function ps(): ProcessResult
    {
        return $this->execute('ps --services');
    }

    public function restart(): ProcessResult
    {
        return $this->execute('restart');
    }

    public function down(): ProcessResult
    {
        return $this->execute('down');
    }

    public function downWithVolumes(): ProcessResult
    {
        return $this->execute('down', '--volumes');
    }

    public function run(string $command): ProcessResult
    {
        return $this->execute('run', $command);
    }

    public function exec(string $command): ProcessResult
    {
        return $this->execute('exec', $command);
    }

    public function container(string $name): self
    {
        $this->container = $name;

        return $this;
    }

    public function logs(array $names): ProcessResult
    {
        return $this->execute('logs', collect($names)->join(' '));
    }

    private function execute(string $action, ?string $internalCommand = null): ProcessResult
    {
        $command = collect(['docker', 'compose']);

        if ($this->location) {
            $command->push('--file');
            $command->push($this->location.DIRECTORY_SEPARATOR.'docker-compose.yml');
        }

        $command->push($action);

        if ($this->follow) {
            $command->push('--follow');
        }

        if ($this->tail) {
            $command->push('--tail');
            $command->push($this->tail);
        }

        if ($this->remove) {
            $command->push('--rm');
        }

        if ($this->detach) {
            $command->push('--detach');
        }

        if ($this->interactive) {
            $command->push('-T');
        }

        if ($this->container) {
            $command->push($this->container);
        }

        if ($internalCommand) {
            $command->push($internalCommand);
        }

        return Process::forever()
            ->when($this->tty, fn ($process) => $process->tty())
            ->run($command->filter()->join(' '));
    }

    public static function boot(): self
    {
        return new self();
    }
}
