<?php

namespace App\Services\Commons\Docker;

use App\Services\CommandService;
use Illuminate\Contracts\Process\ProcessResult;

final class DockerComposeService extends CommandService
{
    private false|string $service = false;

    public function location(string $location): self
    {
        $this->process_path = $location;

        return $this;
    }

    public function interactive(): self
    {
        $this->process_tty = true;

        return $this;
    }

    public function service(string $service): self
    {
        $this->service = $service;

        return $this;
    }

    public function up(bool $detach = true): ProcessResult
    {
        $command = 'docker compose up';

        if ($detach) {
            $command .= ' -d';
        }

        if ($this->service) {
            $command .= $this->service;
        }

        return $this->execute($command);
    }

    public function down(bool $volumes = false): ProcessResult
    {
        $command = 'docker compose down --remove-orphans';

        if ($volumes) {
            $command .= ' --volumes';
        }

        return $this->execute($command);
    }

    public function build(): ProcessResult
    {
        $command = 'docker compose build';

        if ($this->service) {
            $command .= $this->service;
        }

        return $this->execute($command);
    }

    public function exec(string $command): ProcessResult
    {
        return $this->execute("docker compose exec {$this->service} {$command}");
    }

    public function logs(bool $follow = false, int $tail = 0): ProcessResult
    {
        $command = 'docker compose logs';

        if ($follow) {
            $command .= ' -f';
        }

        if ($tail) {
            $command .= " --tail={$tail}";
        }

        if ($this->service) {
            $command .= " {$this->service}";
        }

        return $this->execute($command);
    }

    public function pull(): ProcessResult
    {
        return $this->execute('docker compose pull');
    }

    public function ps(): ProcessResult
    {
        return $this->execute('docker compose ps');
    }

    public function restart(): ProcessResult
    {
        return $this->execute('docker compose restart');
    }

    public function stop(): ProcessResult
    {
        return $this->execute('docker compose stop');
    }

    public function start(): ProcessResult
    {
        return $this->execute('docker compose start');
    }

    public function run(string $command): ProcessResult
    {
        return $this->execute("docker compose run --rm {$this->service} {$command}");
    }
}
