<?php

namespace App\Services;

use Bitbucket\Api\Repositories\Workspaces\Refs;
use Bitbucket\Client;
use Bitbucket\ResultPager;
use GrahamCampbell\Bitbucket\Facades\Bitbucket;
use Http\Client\Exception as HttpException;

final class BitbucketService
{
    private Client $client;

    private ResultPager $resultPager;

    protected function __construct()
    {
        $this->client = Bitbucket::connection();
        $this->resultPager = new ResultPager($this->client);
    }

    public static function boot(): BitbucketService
    {
        return new self();
    }

    public function isRepositoryExists(string $repository): bool
    {

        try {
            $this->client->repositories()->workspaces('maecia')->refs($repository)->list();

            return true;
        } catch (HttpException) {
            return false;
        }
    }

    public function createRepository(string $repository, string $project = 'SID2_INSTANCES'): bool
    {
        try {
            $this->client->repositories()->workspaces('maecia')->create($repository, [
                'project' => ['key' => $project],
                'is_private' => true
            ]);

            return true;
        } catch (HttpException) {
            return false;
        }
    }

    public function enablePipeline(string $repository): bool
    {
        try {
            $result = $this->client->repositories()->workspaces('maecia')->pipelinesConfig($repository)->update([
                'enabled' => true,
            ]);

            return $result['enabled'] === true;
        } catch (HttpException) {
            return false;
        }
    }

    public function disablePipeline(string $repository): bool
    {
        try {
            $result = $this->client->repositories()->workspaces('maecia')->pipelinesConfig($repository)->update([
                'enabled' => false,
            ]);

            return $result['enabled'] === false;
        } catch (HttpException) {
            return false;
        }
    }

    public function getBranchRestriction(string $repository, string $branch): ?array
    {
        try {
            $result = dd($this->client->repositories()->workspaces('maecia')->branchRestrictions($repository)->list());

            return collect($result)->first(fn($item) => $item['pattern'] === $branch);
        } catch (HttpException) {
            return null;
        }
    }

    public function setBranchRestriction(string $repository, string $branch, string $type = 'require_pullrequest'): bool
    {
        try {
            $result = $this->client->repositories()->workspaces('maecia')->branchRestrictions($repository)->create([
                'kind' => $type,
                'pattern' => $branch,
            ]);

            return $result['kind'] === $type;
        } catch (HttpException) {
            return false;
        }
    }
}
