<?php

namespace App\Commands\Initialization;

use App\Actions\Project\Database\CreateDatabaseAction;
use App\Actions\Project\Database\DeleteDatabaseAction;
use App\Actions\Project\Database\ImportDatabaseAction;
use App\Actions\Project\Status\PullProjectAction;
use App\Actions\Project\Status\StartProjectAction;
use App\Actions\Project\Tasks\RunArtisanAction;
use App\Actions\Project\Tasks\RunComposerAction;
use App\Actions\Project\Workers\StartWorkerAction;
use App\Actions\Project\Workers\StopWorkerAction;
use App\Services\CpService;
use App\Services\GitService;
use App\Services\NpmService;
use LaravelZero\Framework\Commands\Command;

class ProjectInstall extends Command
{
    /**
     * The signature of the command.
     *
     * @var string
     */
    protected $signature = 'install {project-code}';

    /**
     * The description of the command.
     *
     * @var string
     */
    protected $description = 'Install an Inside project';

    public function handle(): void
    {
        $code = $this->argument('project-code');
        $directory = config('inside.local.home').DIRECTORY_SEPARATOR.$code;

        $this->title('Installation project '.$code);

        /*--------------------------------------------------------------------------
        | Git clone
        ---------------------------------------------------------------------------*/
        output()->step('Cloning project', 1, 10);
        $result = GitService::clone("git@bitbucket.org:maecia/{$code}.git", $directory);
        output()->debug($result);

        match (true) {
            $result->seeInErrorOutput('fatal: destination path') => output()->warning('GIT', 'Project already cloned!'),
            $result->seeInErrorOutput('You may not have access to this repository ') => output()->kill('GIT', 'Repository not found or access denied!'),
            $result->successful() => output()->success('GIT', "Project cloned in {$directory}!"),
            $result->failed() => output()->kill('GIT', 'Project failed to clone!'),
        };

        /*--------------------------------------------------------------------------
        | Copy .env
        ---------------------------------------------------------------------------*/
        $result = CpService::boot()->from($directory.DIRECTORY_SEPARATOR.'.env.example')->to($directory.DIRECTORY_SEPARATOR.'.env')->copy();
        output()->debug($result);

        match (true) {
            $result->successful() => output()->success('ENV', 'Environment file copied!'),
            $result->failed() => output()->kill('ENV', 'Environment file failed to copy!'),
        };

        /*--------------------------------------------------------------------------
        | Docker pull
        ---------------------------------------------------------------------------*/
        output()->step('Docker compose pull', 2, 10);
        (new PullProjectAction())->pull($directory);

        /*--------------------------------------------------------------------------
        | Composer install
        ---------------------------------------------------------------------------*/
        output()->step('Composer install', 3, 10);
        $result = (new RunComposerAction())->run('install --no-scripts', $directory);

        match (true) {
            $result->seeInErrorOutput('is not recognized as an internal or external command') => output()->kill('COMPOSER', 'Docker is not installed!'),
            $result->successful() => output()->success('COMPOSER', 'Composer dependencies installed!'),
            $result->failed() => output()->kill('COMPOSER', 'Composer dependencies failed to install!'),
        };

        /*--------------------------------------------------------------------------
        | Destroy existing database
        ---------------------------------------------------------------------------*/
        output()->step('Dumping Database', 4, 10);
        (new DeleteDatabaseAction())->delete($directory);

        /*--------------------------------------------------------------------------
        | Recreate database
        ---------------------------------------------------------------------------*/
        (new CreateDatabaseAction())->create($directory);

        /*--------------------------------------------------------------------------
        | Import database from remote server
        ---------------------------------------------------------------------------*/
        (new ImportDatabaseAction())->import(
            directory: $directory,
            host: config('inside.databases.dev-mysql.host'),
            user: config('inside.servers.dev.username'),
            databaseUser: config('inside.databases.dev-mysql.username'),
            databasePassword: config('inside.databases.dev-mysql.password')
        );

        /*--------------------------------------------------------------------------
        | Install npm dependencies
        ---------------------------------------------------------------------------*/
        output()->step('Npm install & build front', 5, 10);
        $result = NpmService::boot()->location($directory)->tty()->install();
        output()->debug($result);

        match (true) {
            $result->successful() => output()->success('NPM', 'Npm dependencies installed!'),
            $result->failed() => output()->kill('NPM', 'Npm dependencies failed to install!'),
        };

        /*--------------------------------------------------------------------------
        | Build npm dependencies
        ---------------------------------------------------------------------------*/
        $result = NpmService::boot()->location($directory)->tty()->build();
        output()->debug($result);

        match (true) {
            $result->successful() => output()->success('NPM', 'Npm dependencies installed!'),
            $result->failed() => output()->kill('NPM', 'Npm dependencies failed to install!'),
        };

        /*--------------------------------------------------------------------------
        | Start containers
        ---------------------------------------------------------------------------*/
        output()->step('Start containers', 6, 10);
        (new StartProjectAction())->start($directory);

        /*--------------------------------------------------------------------------
        | Inside Install
        ---------------------------------------------------------------------------*/
        output()->step('Initialize project', 7, 10);
        $result = (new RunArtisanAction())->run('inside:install', $directory);

        match (true) {
            $result->successful() => output()->success('ARTISAN', 'Inside installed!'),
            $result->failed() => output()->kill('ARTISAN', 'Inside failed to install!'),
        };

        /*--------------------------------------------------------------------------
        | Inside Update
        ---------------------------------------------------------------------------*/
        $result = (new RunArtisanAction())->run('inside:update', $directory);

        match (true) {
            $result->successful() => output()->success('ARTISAN', 'Inside updated!'),
            $result->failed() => output()->kill('ARTISAN', 'Inside failed to update!'),
        };

        /*--------------------------------------------------------------------------
        | Generate app key
        ---------------------------------------------------------------------------*/
        $result = (new RunArtisanAction())->run('key:generate', $directory);

        match (true) {
            $result->successful() => output()->success('ARTISAN', 'App key generated!'),
            $result->failed() => output()->kill('ARTISAN', 'App key failed to generate!'),
        };

        /*--------------------------------------------------------------------------
        | Models rebuild
        ---------------------------------------------------------------------------*/
        $result = (new RunArtisanAction())->run('models:rebuild', $directory);

        match (true) {
            $result->successful() => output()->success('ARTISAN', 'Models rebuilt!'),
            $result->failed() => output()->kill('ARTISAN', 'Models failed to rebuild!'),
        };

        /*--------------------------------------------------------------------------
        | Restart workers
        ---------------------------------------------------------------------------*/
        output()->step('Restart workers', 8, 10);
        (new StopWorkerAction())->stop('queue-indexation', $directory);
        (new StopWorkerAction())->stop('queue-images', $directory);
        (new StopWorkerAction())->stop('queue-default', $directory);
        (new StopWorkerAction())->stop('queue-lts', $directory);
        (new StartWorkerAction())->start('queue-indexation', $directory);
        (new StartWorkerAction())->start('queue-images', $directory);
        (new StartWorkerAction())->start('queue-default', $directory);
        (new StartWorkerAction())->start('queue-lts', $directory);

        /*--------------------------------------------------------------------------
        | Initialize indexation
        ---------------------------------------------------------------------------*/
        output()->step('Initialize indexation', 9, 10);
        $result = (new RunArtisanAction())->run('index:rebuild', $directory);

        match (true) {
            $result->successful() => output()->success('ARTISAN', 'Indexation initialized!'),
            $result->failed() => output()->kill('ARTISAN', 'Indexation failed to initialize!'),
        };

        /*--------------------------------------------------------------------------
        | Get links
        ---------------------------------------------------------------------------*/
        output()->step('links', 10, 10);
        output()->success('WEB', "https://{$code}.docker.localhost");
        output()->success('ADMINER', "https://{$code}-adminer.docker.localhost");
        output()->success('MAIL', "https://{$code}-mail.docker.localhost");
        output()->success('SOLR', "https://{$code}-solr.docker.localhost");
    }
}
