<?php

namespace App\Commands\Creation;

use App\Services\BitbucketService;
use App\Services\CpService;
use App\Services\GitService;
use App\Services\RmService;
use App\Services\SedService;
use App\Traits\DirectoryPathfinder;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Validator;
use LaravelZero\Framework\Commands\Command;
use Symfony\Component\Finder\SplFileInfo;

class CreateProject extends Command
{
    use DirectoryPathfinder;

    /**
     * The signature of the command.
     *
     * @var string
     */
    protected $signature = 'create:repository:back {code}';

    /**
     * The description of the command.
     *
     * @var string
     */
    protected $description = 'Create a new back repository';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $code = $this->argument('code');

        BitbucketService::boot()->getBranchRestriction("soge-back", 'develop');

        $this->title('Creating back repository: ' . $code.'-back');

        /**
         * Step 1: Check if this project code is available
         */
        output()->step('Check if this project code is available', 1, 10);
        $validator = Validator::make(['name' => $code], [
            'name' => 'required|string|min:4|max:4',
        ]);

        match ($validator->fails()) {
            false => output()->success('CODE', "Project code is valid"),
            true => output()->kill('CODE', 'Invalid project code (4 characters)')
        };

        match (BitbucketService::boot()->isRepositoryExists("$code-back")) {
            true => output()->kill('BITBUCKET', 'Repository already exists on Bitbucket'),
            false => output()->success('BITBUCKET', "Repository name is available")
        };

        /**
         * Step 2: Create temporary directory
         */
        output()->step('Creating temporary directory', 2, 10);
        $temporary_path = $this->createPath('/tmp', $code . '-back');
        output()->success('FOLDER', "Temporary directory created at $temporary_path");

        /**
         * Step 3: Copying template files
         */
        output()->step('Copying template files', 3, 10);
        CpService::boot()->from($this->backTemplatePath())->to($temporary_path)->copy();
        output()->success('COPY', "Template files copied to $temporary_path");

        /**
         * Step 4: Replacing template blank with code
         */
        output()->step('Replacing template blank with code', 4, 10);
        collect(File::allFiles($temporary_path))->each(function (SplFileInfo $file) use ($code) {
            SedService::boot()->input($file->getRealPath())->search('BLANK')->replace(strtoupper($code))->execute();
            SedService::boot()->input($file->getRealPath())->search('blank')->replace(strtolower($code))->execute();
        });
        output()->success('REPLACE', "Template files replaced with $code");

        /**
         * Step 5: Creating repository on Bitbucket
         */
        output()->step('Creating repository on Bitbucket', 5, 10);
        match (BitbucketService::boot()->createRepository("$code-back")) {
            false => output()->kill('BITBUCKET', 'Failed to create repository on Bitbucket'),
            true => output()->success('BITBUCKET', "Repository created on Bitbucket"),
        };

        /**
         * Step 6: Create master and pushing initial commit to Bitbucket
         */
        output()->step('Create master and pushing initial commit to Bitbucket', 6, 10);
        $git = GitService::on($temporary_path);

        $result = $git->init();
        output()->debug($result);
        match(true) {
            $result->failed() => output()->kill('GIT', 'Failed to initialize git repository'),
            default => output()->success('GIT', 'Git repository initialized'),
        };

        $result = $git->remote('origin', "git@bitbucket.org:maecia/$code-back.git");
        output()->debug($result);
        match(true) {
            $result->failed() => output()->kill('GIT', 'Failed to add remote origin'),
            default => output()->success('GIT', 'Remote origin added'),
        };

        $result = $git->checkout('master');
        output()->debug($result);
        match(true) {
            $result->failed() => output()->kill('GIT', 'Failed to checkout master branch'),
            default => output()->success('GIT', 'Checked out master branch'),
        };

        $result = $git->add('.');
        output()->debug($result);
        match(true) {
            $result->failed() => output()->kill('GIT', 'Failed to add files to git'),
            default => output()->success('GIT', 'Files added to git'),
        };

        $result = $git->commit('Initial commit');
        output()->debug($result);
        match(true) {
            $result->failed() => output()->kill('GIT', 'Failed to commit files'),
            default => output()->success('GIT', 'Files committed'),
        };

        $result = $git->push('origin', 'develop');
        output()->debug($result);
        match(true) {
            $result->failed() => output()->kill('GIT', 'Failed to push to origin'),
            default => output()->success('GIT', 'Pushed to origin'),
        };

        /**
         * Step 7: Create develop and pushing initial commit to Bitbucket
         */
        output()->step('Create develop and pushing initial commit to Bitbucket', 6, 10);
        $result = $git->checkout('develop');
        output()->debug($result);
        match(true) {
            $result->failed() => output()->kill('GIT', 'Failed to checkout develop branch'),
            default => output()->success('GIT', 'Checked out develop branch'),
        };

        $result = $git->push('origin', 'develop');
        output()->debug($result);
        match(true) {
            $result->failed() => output()->kill('GIT', 'Failed to push to origin'),
            default => output()->success('GIT', 'Pushed to origin'),
        };

        /**
         * Step 8: Removing temporary directory
         */
        output()->step('Removing temporary directory', 8, 10);
        $result = RmService::boot()->path($temporary_path)->recursive()->force()->execute();
        output()->debug($result);
        match(true) {
            $result->failed() => output()->kill('REMOVE', 'Failed to remove temporary directory'),
            default => output()->success('REMOVE', 'Temporary directory removed'),
        };

        /**
         * Step 9: Enable pipeline on Bitbucket
         */
        output()->step('Enable pipeline on Bitbucket', 9, 10);
        match (BitbucketService::boot()->enablePipeline("$code-back")) {
            false => output()->kill('BITBUCKET', 'Failed to enable pipeline on Bitbucket'),
            true => output()->success('BITBUCKET', 'Pipeline enabled on Bitbucket'),
        };

        return 1;
    }
}
