<?php

namespace Inside\Authentication\Windows\Services;

use Inside\Authentication\Windows\Contracts\AuthenticationWindows;

/**
 * windowsAuth.class.php
 *
 * Get Windows Credentials from a IIS server and a client connected through
 * Active Directory Natively supported by IE & Chrome on Windows. For others, a
 * user/password prompt will display
 *
 * @category   Authentication
 * @package    IIS
 * @author     Charles Bourgeaux
 * @copyright  2016 Charles Bourgeaux - Maecia
 * @license    http://www.php.net/license/3_0.txt  PHP License 3.0
 * @version    1.0.0
 * @link       https://github.com/charlyie/windows-credentials-active-directory.git
 */
class AuthenticationWindowsService implements AuthenticationWindows
{
    public ?string $authenticationMethod = null;

    public ?string $domain = null;

    public ?string $user = null;

    private ?int $authenticationOffset = null;

    private ?string $authorizationToken = null;

    public bool $initialized = false;

    public array $errors = [];

    /**
     * Check user's headers and determine Authentication method
     */
    private function checkHeaders(): void
    {
        $clientHeaders = apache_request_headers();

        if (!$clientHeaders) {
            $clientHeaders = [];
        }

        // NTLM does not work through a proxy
        if (@$_SERVER['HTTP_VIA'] != null) {
            $this->declareError('101', 'Windows authentication cannot work through a proxy.');
            return;
        }

        //if authorization header does not exist, throw to the client a 401 ERROR
        if (!isset($clientHeaders['Authorization']) && !isset($_SERVER['AUTH_USER'])) {
            $this->declareError('102', 'Authorization entry not found in client headers.');
        }

        if (isset($clientHeaders['Authorization'])) {
            $this->authorizationToken = $clientHeaders['Authorization'];
        }

        if (str_starts_with($this->authorizationToken ?? '', 'NTLM ')) {
            $this->authenticationMethod = 'NTLM';
            $this->authenticationOffset = 5;
        } elseif (str_starts_with($this->authorizationToken ?? '', 'Negotiate ')) {
            $this->authenticationMethod = 'NEGOTIATE';
            $this->authenticationOffset = 10;
        } elseif (isset($_SERVER['AUTH_USER']) && $_SERVER['AUTH_USER']) {
            $this->authenticationMethod = 'SERVER';
            $this->authenticationOffset = 0;
        }
    }

    /**
     * Retrieve user from authentication
     */
    public function getUser(): ?string
    {
        if (!$this->authenticationMethod) {
            $this->getCredentials();
        }

        return $this->user;
    }

    /**
     * Retrieve domain from authentication
     */
    public function getDomain(): ?string
    {
        if (!$this->authenticationMethod) {
            $this->getCredentials();
        }

        return $this->domain;
    }

    /**
     * Provides user's credentials
     */
    public function getCredentials(): false|AuthenticationWindows
    {
        $this->initialized = true;

        $this->checkHeaders();
        if (!$this->authenticationMethod) {
            $this->declareError('103', 'Unknown authentication method. Should be NTLM or Negociate (Kerberos).');
            return false;
        }

        $force_server_auth = false;

        if ($this->authenticationMethod != 'SERVER' && is_string($this->authorizationToken) && is_int($this->authenticationOffset)) {
            $token64 = base64_decode(substr(
                $this->authorizationToken,
                $this->authenticationOffset
            )); // get Base64-encoded type1 message

            if (ord($token64[8]) == 1) {
                /*$retAuth = 'NTLMSSP' . chr(000) . chr(002) . chr(000)
                . chr(000) . chr(000) . chr(000) . chr(000) . chr(000);
                $retAuth .= chr(000) . chr(040) . chr(000) . chr(000)
                . chr(000) . chr(001) . chr(130) . chr(000) . chr(000);
                $retAuth .= chr(000) . chr(002) . chr(002) . chr(002)
                . chr(000) . chr(000) . chr(000) . chr(000) . chr(000);
                $retAuth .= chr(000) . chr(000) . chr(000) . chr(000)
                . chr(000) . chr(000) . chr(000);
                $retAuth64 = base64_encode($retAuth);
                $retAuth64 = trim($retAuth64);

                $this->declareError('104', 'Authentication needs further informations.');

                header('HTTP/1.1 401 Unauthorized'); // send new header
                header('WWW-Authenticate: NTLM ' . $retAuth64); // need additionnal authentication
                exit;*/

                $force_server_auth = true;
            } elseif (ord($token64[8]) == 3) {
                $lenght_domain = (ord($token64[31]) * 256 + ord($token64[30])); // domain length
                $offset_domain = (ord($token64[33]) * 256 + ord($token64[32])); // domain position
                $this->domain = str_replace("\0", '', substr(
                    $token64,
                    $offset_domain,
                    $lenght_domain
                )); // extracting domain

                $lenght_login = (ord($token64[39]) * 256 + ord($token64[38])); // user length
                $offset_login = (ord($token64[41]) * 256 + ord($token64[40])); // user position
                $this->user = str_replace("\0", '', substr($token64, $offset_login, $lenght_login)); // extracting user

                if (empty($this->domain)) {
                    $this->declareError('105', 'Cannot guess
                     Active Directory Domain. Maybe a bad environment
                     (OS/Browser) ?');
                }
                if (!empty($this->user)) {
                    return $this;
                }
            }
        }
        if ($this->authenticationMethod == 'SERVER' || $force_server_auth) {
            $login_data = explode('\\', $_SERVER['AUTH_USER']);

            if (sizeof($login_data) < 2) {
                $this->declareError('106', 'Cannot find user with domain from SERVER method.');
                return false;
            }

            $this->user = $login_data[sizeof($login_data) - 1];
            $delimiter = null;

            foreach ($login_data as $key => $value) {
                if ($key == (sizeof($login_data) - 1)) {
                    continue;
                }

                if ($this->domain) {
                    $delimiter = '\\';
                }

                $this->domain .= $delimiter . $value;
            }
            return $this;
        }

        $this->declareError('107', 'Cannot decode properly authorization token');
        return false;
    }

    /**
     * Internal use : for error declaration purposes
     */
    public function declareError(string $code, string $description): void
    {
        $this->errors[] = ['code' => $code, 'long' => $description];
    }

    /**
     * Internal use : to display errors
     */
    public function getErrors(string $format = 'array'): array|string|null
    {
        $errorString = null;

        if (!$this->initialized) {
            $this->declareError('001', 'getCredentials or getUser methods have to be launched BEFORE error detection');
        }

        if (sizeof($this->errors) == 0) {
            return null;
        }

        if ($format == 'array') {
            return $this->errors;
        }

        foreach ($this->errors as $error) {
            $errorString .= 'Error #' . $error['code'] . ' : ' . $error['long'] . '<br>';
        }

        return $errorString;
    }
}
