<?php

namespace Inside\Authentication\Windows\Http\Controllers;

use Illuminate\Auth\AuthenticationException;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Str;
use Inside\Authentication\Contracts\Authentication;
use Inside\Authentication\Windows\Contracts\AuthenticationWindows as AuthenticationWindowsContract;
use Inside\Authentication\Models\User;
use Inside\Authentication\Windows\Facades\AuthenticationWindows;
use Inside\Content\Models\Contents\Users;
use Inside\Kernel\Authentication\Actions\SetMagicCookie;
use Inside\Kernel\Authentication\SingleSignOnProviderEnum;

/**
 * Authentication windows controller.
 *
 * @category Class
 * @package  Inside\Authentication\Windows\Controllers\AuthenticationWindowsController
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class AuthenticationWindowsController
{
    protected int $lifetime = 1;

    /**
     * This force to show kerberos popup
     *
     * @throws AuthenticationException
     */
    public function showKerberosLogin(
        Authentication $authentication,
        AuthenticationWindowsContract $authenticationWindows
    ): JsonResponse {
        // Kerberos connection OK!!
        if (isset($_SERVER['AUTH_USER']) && !empty($_SERVER['AUTH_USER'])) {
            $authenticationWindows->getCredentials();

            if (!$authenticationWindows->getErrors()) {
                $name = $authenticationWindows->getUser();
                if ($name !== null) {
                    if ($this->loginUser($authentication, $name)) {
                        $secure = Str::startsWith(env('APP_URL'), 'https://');

                        return response()->json(
                            [
                                // Set redirection with api token to start front session
                                'redirect' => url('/external-login', [], $secure),
                            ]
                        );
                    }
                }
            }
        }
        // Force kerberos negotiation token
        throw new AuthenticationException();
    }

    /**
     * Log the user in.
     */
    public function login(Authentication $authentication, AuthenticationWindowsContract $authenticationWindows): mixed
    {
        $secure = Str::startsWith(env('APP_URL'), 'https://');
        $name = '';

        if (isset($_SERVER['AUTH_USER']) && !empty($_SERVER['AUTH_USER'])) {
            $authenticationWindows->getCredentials();

            if (!$authenticationWindows->getErrors()) {
                $name = $authenticationWindows->getUser();
            }
        } else {
            if (isset($_SERVER['REMOTE_USER']) && !empty($_SERVER['REMOTE_USER'])) {
                $remoteUser = explode("@", $_SERVER['REMOTE_USER']);
                $name = $remoteUser[0];
            } elseif (isset($_SERVER['PHP_AUTH_USER'])) {
                $name = $_SERVER['PHP_AUTH_USER'];
            } else {
                // Did we already asked kerberos for a token? ! ( avoid loop )
                if (Cookie::has(md5('askedkerberostoken'))) {
                    // Cookie should die by itself after lifetime minute.s, but we don't trust others!! let's check lifetime
                    $born = Carbon::createFromTimestamp((int) Cookie::get(md5('askedkerberostoken')));
                    if ($born->diffInMinutes() <= $this->lifetime) {
                        // We did everything we could to log using windows authentication
                        // Let's fallback to inside login
                        return redirect(Config::get('app.url').'/login')->withCookie(
                            Cookie::forget(md5('askedkerberostoken'))
                        );
                    }
                }
                // Drop a cookie so we can remember for $this->lifetime minute.s that we asked for a kerberos token
                Cookie::queue(Cookie::make(md5('askedkerberostoken'), time(), $this->lifetime));

                // Force kerberos to load REMOTE_USER & token ( we made a special view for that )
                return view(
                    'windows',
                    [
                        'kerberos_url' => route('authentication.windows.kerberos', [], $secure),
                        'login_url' => route('authentication.windows.login', [], $secure),
                    ]
                );
            }
        }

        if (!empty($name)) {
            if ($this->loginUser($authentication, $name)) {
                return redirect('/external-login', 302, [], $secure)
                    ->withCookie(Cookie::forget(md5('askedkerberostoken')));
            }
        }

        return redirect(Config::get('app.url').'/login')
            ->withCookie(Cookie::forget(md5('askedkerberostoken')));
    }

    /**
     * loginUser $userName and get its access token
     */
    protected function loginUser(Authentication $authentication, string $userName): bool
    {
        $discover = env('WINDOWS_AUTH_DISCOVER', 'name');

        if ($discover === 'name') {
            $user = User::where($discover, $userName)->first();
        } else {
            $user = Users::where($discover, $userName)->first();

            if ($user) {
                $user = User::find($user->uuid);
            }
        }

        if ($user instanceof User) {
            (new SetMagicCookie())->execute(request(), $user, SingleSignOnProviderEnum::WINDOWS);

            return true;
        }

        return false;
    }
}
