<?php

namespace Inside\Authentication\SAML\SAML;

use ArrayAccess;
use Exception;
use Illuminate\Support\Arr;
use Inside\Authentication\SAML\Models\User;
use OneLogin\Saml2\Auth;
use OneLogin\Saml2\Error;
use OneLogin\Saml2\ValidationError;
use Psr\Log\InvalidArgumentException;

/**
 * Class Authenticator
 *
 * Warper to OneLogin Saml2 auth
 *
 * @package Inside\Authentication\SAML\SAML
 */
class Authenticator
{
    public function __construct(
        protected Auth $auth,
        protected string $name,
        protected array $config
    ) {
    }


    /**
     * Login via onelogin saml2
     * @throws Error
     */
    public function login(
        ?string $returnTo = null,
        array $parameters = [],
        bool $forceAuthn = false,
        bool $isPassive = false,
        bool $stay = false,
        bool $setNameIdPolicy = true
    ): ?string {
        return $this->auth->login($returnTo, $parameters, $forceAuthn, $isPassive, $stay, $setNameIdPolicy);
    }


    /**
     * Indicates if user is authenticated with Saml
     */
    public function isAuthenticated(): bool
    {
        return $this->auth->isAuthenticated();
    }


    /**
     * Process SAML Login response
     *
     * @throws Error
     * @throws ValidationError
     */
    public function acs(): ?array
    {
        $this->auth->processResponse();

        $errors = $this->auth->getErrors();

        if (! empty($errors)) {
            return $errors;
        }

        if (! $this->auth->isAuthenticated()) {
            return [ 'error' => 'Could not authenticate' ];
        }

        return null;
    }


    /**
     * Logout
     *
     * @param string|null $returnTo
     * @param string|null $nameId
     * @param string|null $sessionIndex
     * @param string|null $nameIdFormat
     *
     * @throws Error
     */
    public function logout(
        ?string $returnTo = null,
        ?string $nameId = null,
        ?string $sessionIndex = null,
        ?string $nameIdFormat = null
    ): void {
        $this->auth->logout($returnTo, [], $nameId, $sessionIndex, false, $nameIdFormat);
    }


    /**
     * Process SAML Logout response
     * @throws Error
     */
    public function sls(bool $retrieveParametersFromServer = false): array
    {
        $this->auth->processSLO(false, null, $retrieveParametersFromServer);

        return $this->auth->getErrors();
    }

    /**
     * GetMetadata from Config
     * @throws Error
     * @throws Exception
     */
    public function getMetadata(): string
    {
        $settings = $this->auth->getSettings();
        $metadata = $settings->getSPMetadata(true);
        $errors   = $settings->validateMetadata($metadata);

        if (empty($errors)) {
            return $metadata;
        } else {
            throw new InvalidArgumentException(
                'Invalid SP metadata: ' . implode(', ', $errors),
                Error::METADATA_SP_INVALID
            );
        }
    }

    public function getLastErrorReason(): ?string
    {
        return $this->auth->getLastErrorReason();
    }

    public function getUser(): User
    {
        return new User($this->auth);
    }

    public function getAttributeWithFriendlyName(string $key): ?array
    {
        return $this->auth->getAttributeWithFriendlyName($key);
    }

    public function getAttributes(): array
    {
        return $this->auth->getAttributes();
    }

    public function getName(): string
    {
        return $this->name;
    }

    public function config(string $key = null, bool|string $default = null): mixed
    {
        if (is_null($key)) {
            return $this->config;
        }

        return Arr::get($this->config, $key, $default);
    }
}
