<?php

namespace Inside\Authentication\SAML\Models;

use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use OneLogin\Saml2\Auth;

/**
 * Class User
 *
 * This is the representation of SAML User
 *
 * @package Inside\Authentication\SAML\Models
 */
class User
{
    protected Collection $attributes;

    public function __construct(
        protected Auth $auth
    ) {
        $this->attributes = $this->getAttributes();
    }

    /**
     * UserId is NameId ( check config )
     *
     * @throws Exception
     */
    public function getUserId(): string
    {
        return $this->auth->getNameId();
    }

    /**
     * Get All attributes asked from config
     */
    protected function getAttributes(): Collection
    {
        // First check if config set attributes
        $config = Config::get('saml2.sp.attributeConsumingService.requestedAttributes', []);
        if (empty($config)) {
            return collect($this->auth->getAttributes()); // No attribute asked
        }

        $attributes = collect();
        foreach ($config as $attribute) {
            /** @var array $values */
            $values = $this->getAttribute($attribute['name']);
            if (count($values) > 0) {
                $attributes[$attribute['friendlyName']] = $values[0];
            }
        }

        return $attributes;
    }

    /**
     * Get One Attribute from auth
     */
    protected function getAttribute(string $name): ?array
    {
        return $this->auth->getAttribute($name);
    }


    /**
     * Get SAML Session ID
     */
    public function getSessionIndex(): ?string
    {
        return $this->auth->getSessionIndex();
    }


    /**
     * Get NameId / Same as getUserId
     */
    public function getNameId(): string
    {
        return $this->auth->getNameId();
    }


    /**
     * Magic getter to access attributes
     *
     * For example, if you asked for email ( friendlyNamed email in config )
     * you can access via $user->email
     */
    public function __get(string $name): mixed
    {
        if ($this->attributes->has($name)) {
            return $this->attributes[$name];
        }

        return null;
    }
}
