<?php

namespace Inside\Authentication\SAML\Http\Controllers;

use Inside\Authentication\SAML\Http\Controllers\Saml2Controller;
use Inside\Authentication\SAML\SAML\AuthenticatorManager;
use Illuminate\Filesystem\Filesystem;
use Illuminate\Http\Request;
use Laravel\Lumen\Routing\Controller;
use Inside\Settings\Models\Setting;
use Illuminate\Http\JsonResponse;
use OneLogin\Saml2\Auth;
use Illuminate\Http\Response;
use Inside\Facades\Http;

class Saml2ConfigController extends Controller
{
    protected Filesystem $filesystem;

    protected string $certsDir;

    public function __construct(Filesystem $filesystem)
    {
        $this->filesystem = $filesystem;
        $this->certsDir = cms_base_path('certs');
    }

    /**
     * Generate SAML key pair and save to certs directory
     */
    public function generateKeys(): JsonResponse
    {
        if (! $this->filesystem->exists($this->certsDir)) {
            $this->filesystem->makeDirectory($this->certsDir, 0700, true);
        }

        $privateKeyFile = $this->certsDir . '/saml.key';
        $certFile = $this->certsDir . '/saml.crt';

        if ($this->filesystem->exists($privateKeyFile) || $this->filesystem->exists($certFile)) {
            return new JsonResponse([
                'status' => 'error',
                'message' => __('saml.keys_already_exist')
            ], 409);
        }

        $privateKey = openssl_pkey_new([
            'private_key_type' => OPENSSL_KEYTYPE_RSA,
            'private_key_bits' => 2048,
        ]);
        if (!$privateKey) {
            return new JsonResponse([
                'status' => 'error',
                'message' => __('saml.key_generation_failed')
            ], 500);
        }

        $dn = [
            "countryName" => "FR",
            "stateOrProvinceName" => "IDF",
            "localityName" => "Paris",
            "organizationName" => "Maecia",
            "organizationalUnitName" => "IT",
            "commonName" => "saml.local",
            "emailAddress" => "technique@maecia.com"
        ];
        $csr = openssl_csr_new($dn, $privateKey, ['digest_alg' => 'sha256']);
        if (!$csr) {
            return new JsonResponse([
                'status' => 'error',
                'message' => __('saml.csr_generation_failed')
            ], 500);
        }

        $x509 = openssl_csr_sign($csr, null, $privateKey, 3652, ['digest_alg' => 'sha256']);
        if (!$x509) {
            return new JsonResponse([
                'status' => 'error',
                'message' => __('saml.certificate_signing_failed')
            ], 500);
        }

        openssl_pkey_export($privateKey, $privateKeyOut);
        $this->filesystem->put($privateKeyFile, $privateKeyOut);

        openssl_x509_export($x509, $certOut);
        $this->filesystem->put($certFile, $certOut);

        return new JsonResponse([
            'status' => 'ok',
            'message' => __('saml.keys_generated_successfully')
        ], 200);
    }

    /**
     * Check if SAML key pair exists in certs directory
     */
    public function keysExist(): JsonResponse
    {
        $privateKeyFile = $this->certsDir . '/saml.key';
        $certFile = $this->certsDir . '/saml.crt';

        $exists = $this->filesystem->exists($privateKeyFile) && $this->filesystem->exists($certFile);

        return new JsonResponse([
            'keys_exist' => $exists,
        ]);
    }

    /**
     * Download SAML metadata XML file
     */
    public function downloadMetadata(): Response
    {
        $saml2Controller = app(Saml2Controller::class);
        $request = request();
        $response = $saml2Controller->metadata($request);
        $metadata = $response->getContent();

        $filename = 'saml-metadata.xml';
        return new Response($metadata, 200, [
            'Content-Type' => 'application/xml',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }

    /**
     * Set federation metadata XML file into settings
     */
    public function setMetadata(Request $request): JsonResponse
    {
        $this->validate($request, [
            'metadata_url' => 'required|url',
        ]);

        Setting::updateOrCreate(
            ['key' => 'saml_federation_metadata_url', 'group' => 'sso_config'],
            ['value' => $request->input('metadata_url')]
        );

        return new JsonResponse([
            'status' => 'ok',
            'message' => __('saml.metadata_saved_successfully')
        ], 200);
    }

    public function checkConfiguration(): JsonResponse
    {
        $privateKeyFile = $this->certsDir . '/saml.key';
        $certFile = $this->certsDir . '/saml.crt';

        if (!file_exists($privateKeyFile) || !file_exists($certFile)) {
            return new JsonResponse([
                'status' => 'error',
                'message' => __('saml.keys_not_found')
            ], 500);
        }

        try {
            $privateKeyContent = file_get_contents($privateKeyFile);
            $certContent = file_get_contents($certFile);

            if ($privateKeyContent === false || $certContent === false) {
                return new JsonResponse([
                    'status' => 'error',
                    'message' => __('saml.keys_read_error')
                ], 500);
            }

            $privateKey = openssl_pkey_get_private($privateKeyContent);
            $cert = openssl_x509_read($certContent);

            if ($privateKey === false || $cert === false) {
                return new JsonResponse([
                    'status' => 'error',
                    'message' => __('saml.key_cert_invalid')
                ], 500);
            }

            if (!openssl_x509_check_private_key($cert, $privateKey)) {
                return new JsonResponse([
                    'status' => 'error',
                    'message' => __('saml.key_cert_mismatch')
                ], 500);
            }
        } catch (\Throwable $e) {
            return new JsonResponse([
                'status' => 'error',
                'message' => __('saml.key_cert_invalid')
            ], 500);
        }

        $metadataUrl = setting('sso_config', 'saml_federation_metadata_url', null);
        if (!$metadataUrl) {
            return new JsonResponse([
                'status' => 'error',
                'message' => __('saml.no_metadata_url_configured')
            ], 500);
        }

        try {
            $response = Http::get($metadataUrl);
            if (!$response->ok()) {
                return new JsonResponse([
                    'status' => 'error',
                    'message' => __('saml.metadata_url_unreachable', ['url' => $metadataUrl])
                ], 500);
            }

            libxml_use_internal_errors(true);
            $xml = simplexml_load_string($response->body());
            if ($xml === false) {
                return new JsonResponse([
                    'status' => 'error',
                    'message' => __('saml.metadata_invalid_xml')
                ], 500);
            }
        } catch (\Throwable $e) {
            return new JsonResponse([
                'status' => 'error',
                'message' => __('saml.metadata_fetch_failed')
            ], 500);
        }

        try {
            $saml2Controller = app(Saml2Controller::class);
            $response = $saml2Controller->metadata(request());
            if ($response->getStatusCode() !== 200) {
                return new JsonResponse([
                    'status' => 'error',
                    'message' => __('saml.sp_metadata_generation_failed')
                ], 500);
            }
        } catch (\Throwable $e) {
            return new JsonResponse([
                'status' => 'error',
                'message' => __('saml.sp_metadata_error')
            ], 500);
        }

        try {
            /** @var \Inside\Authentication\SAML\SAML\Authenticator|null $authenticator */
            $authenticator = app(AuthenticatorManager::class)->get('default');

            if (!$authenticator) {
                return new JsonResponse([
                    'status' => 'error',
                    'message' => __('saml.authenticator_not_registered')
                ], 500);
            }
        } catch (\Throwable $e) {
            return new JsonResponse([
                'status' => 'error',
                'message' => __('saml.authenticator_error')
            ], 500);
        }

        return new JsonResponse([
            'status' => 'ok',
            'message' => __('saml.configuration_valid')
        ], 200);
    }
}
