<?php

namespace Inside\Authentication\SAML\SAML;

use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Str;
use OneLogin\Saml2\Auth;
use OneLogin\Saml2\Error;
use OneLogin\Saml2\Utils;
use Illuminate\Support\Facades\Cache;

class AuthenticatorManager
{
    protected Collection $authenticators;

    public function __construct()
    {
        $this->authenticators = collect();
    }

    /**
     * @throws Error
     * @throws Exception
     */
    public function registerAuthenticator(array $config, string $authenticatorName = 'default'): void
    {
        $config = $this->formatConfig($authenticatorName, $config);
        $auth = new Auth($config);
        $authenticator = new Authenticator($auth, $authenticatorName, $config);

        $this->authenticators->put($authenticatorName, $authenticator);
    }

    public function getDefault(): ?Authenticator
    {
        return $this->get('default');
    }

    public function exists(string $name): bool
    {
        return $this->authenticators->has($name);
    }

    public function get(string $name): ?Authenticator
    {
        return $this->authenticators->get($name);
    }

    /**
     * Read Private Key file
     *
     * @throws Exception
     */
    protected function extractPkeyFromFile(string $path): string
    {
        $res = openssl_get_privatekey($path);
        if (empty($res)) {
            throw new Exception('Could not read private key-file at path \''.$path.'\'');
        }
        openssl_pkey_export($res, $pkey);

        return $this->extractOpensslString($pkey, 'PRIVATE KEY');
    }

    /**
     * Read public certificate
     * @throws Exception
     */
    protected function extractCertFromFile(string $path): string
    {
        $res = openssl_x509_read((string) file_get_contents($path));
        if (empty($res)) {
            throw new Exception('Could not read X509 certificate-file at path \''.$path.'\'');
        }
        openssl_x509_export($res, $cert);

        return $this->extractOpensslString($cert, 'CERTIFICATE');
    }


    /**
     * Extract Open SSL string
     */
    protected function extractOpensslString(string $keyString, string $delimiter): string
    {
        $keyString = str_replace(["\r", "\n"], "", $keyString);
        $regex = '/-{5}BEGIN(?:\s|\w)+'.$delimiter.'-{5}\s*(.+?)\s*-{5}END(?:\s|\w)+'.$delimiter.'-{5}/m';
        preg_match($regex, $keyString, $matches);

        return empty($matches[1]) ? '' : $matches[1];
    }

    protected function getBindingFromArray(array $xml, string $singleService, array $config): string
    {
        $service = Arr::get($xml, 'IDPSSODescriptor.' . $singleService);

        if (!$service) {
            return "";
        }

        //try to get the right binding if we are in an array
        $first = Arr::first($service, function ($value, $key) use ($config) {
            return $value['@attributes']['Binding'] === $config['idp']['binding'];
        });
        if ($first) {
            return Arr::get($first, '@attributes.Location');
        } else {
            // get the first one if we can't find the binding
            return Arr::get($xml, 'IDPSSODescriptor.' . $singleService . '.0.@attributes.Location');
        }
    }

    /**
     * @throws Exception
     */
    protected function formatIDPConfig(string $authenticatorName, array $config): array
    {
        $multiCertificate = false;
        if ($config['idp']['metadata']) {
            $key = md5(sprintf(
                '%s-%s-saml-metadata',
                config('app.code'),
                $authenticatorName
            ));
            if (!Cache::has($key)) {
                $metadata = str_starts_with($config['idp']['metadata'], '<?xml')
                    ? $config['idp']['metadata']
                    : file_get_contents($config['idp']['metadata']);

                if (! $metadata) {
                    throw new Exception("error while getting metadata file content");
                }
                $content = (string) preg_replace("/(<\/?)([a-zA-Z]{2}:)/", '${1}', $metadata);
                $xmlPage = simplexml_load_string($content);
                //transform std::class to array
                $xml = json_decode((string) json_encode($xmlPage), true);
                Cache::put($key, $xml, $config['idp']['cacheTime']);
            } else {
                $xml = Cache::get($key);
            }
            $certificate = Arr::get($xml, 'IDPSSODescriptor.KeyDescriptor.KeyInfo.X509Data.X509Certificate');
            if (!$certificate) {
                $keyDescriptor = Arr::get($xml, 'IDPSSODescriptor.KeyDescriptor');
                $multiCertificate = true;
                $certificate = [];
                foreach ($keyDescriptor as $value) {
                    if ($value['@attributes']['use'] === "signing") {
                        $certificate[] = Arr::get($value, 'KeyInfo.X509Data.X509Certificate');
                    }
                }
            }
            $entityId = Arr::get($xml, '@attributes.entityID');
            $ssoService = Arr::get($xml, 'IDPSSODescriptor.SingleSignOnService.@attributes.Location');
            if (!$ssoService) {
                $ssoService = $this->getBindingFromArray($xml, 'SingleSignOnService', $config);
            }
            $logoutService = Arr::get($xml, 'IDPSSODescriptor.SingleLogoutService.@attributes.Location');
            if (!$logoutService) {
                $logoutService = $this->getBindingFromArray($xml, 'SingleLogoutService', $config);
            }

            if ($certificate && $multiCertificate) {
                $config['idp']['x509certMulti']['signing'] = $certificate;
            } elseif ($certificate) {
                $config['idp']['x509cert'] = $certificate;
            }
            if ($entityId) {
                $config['idp']['entityId'] = $entityId;
            }
            if ($ssoService) {
                $config['idp']['singleSignOnService']['url'] = $ssoService;
            }
            if (env('SAML2_SINGLE_SIGN_ON_SERVICE_URI') !== null) {
                $config['idp']['singleSignOnService']['url'] = env('SAML2_SINGLE_SIGN_ON_SERVICE_URI');
            }
            if ($logoutService) {
                $config['idp']['singleLogoutService']['url'] = $ssoService;
            }
            if (env('SAML2_SINGLE_LOGOUT_SERVICE_URI') !== null) {
                $config['idp']['singleLogoutService']['url'] = env('SAML2_SINGLE_LOGOUT_SERVICE_URI');
            }
        } else {
            if ($authenticatorName !== 'default') {
                return $config;
            }

            $idpHost = env('SAML2_IDP_HOST', 'http://localhost:8000/simplesaml');
            $config['idp']['entityId'] = env('SAML2_IDP_ENTITYID', $idpHost . '/saml2/idp/metadata.php');
            $config['idp']['singleLogoutService']['url'] = env('SAML2_SINGLE_LOGOUT_SERVICE_URI', $idpHost . '/saml2/idp/SingleLogoutService.php');
            $config['idp']['singleSignOnService']['url'] = env('SAML2_SINGLE_SIGN_ON_SERVICE_URI', $idpHost . '/saml2/idp/SSOService.php');
            $config['idp']['x509cert'] = env(
                'SAML2_IDP_x509',
                'MIID / TCCAuWgAwIBAgIJAI4R3WyjjmB1MA0GCSqGSIb3DQEBCwUAMIGUMQswCQYDVQQGEwJBUjEVMBMGA1UECAwMQnVlbm9zIEFpcmVzMRUwEwYDVQQHDAxCdWVub3MgQWlyZXMxDDAKBgNVBAoMA1NJVTERMA8GA1UECwwIU2lzdGVtYXMxFDASBgNVBAMMC09yZy5TaXUuQ29tMSAwHgYJKoZIhvcNAQkBFhFhZG1pbmlAc2l1LmVkdS5hcjAeFw0xNDEyMDExNDM2MjVaFw0yNDExMzAxNDM2MjVaMIGUMQswCQYDVQQGEwJBUjEVMBMGA1UECAwMQnVlbm9zIEFpcmVzMRUwEwYDVQQHDAxCdWVub3MgQWlyZXMxDDAKBgNVBAoMA1NJVTERMA8GA1UECwwIU2lzdGVtYXMxFDASBgNVBAMMC09yZy5TaXUuQ29tMSAwHgYJKoZIhvcNAQkBFhFhZG1pbmlAc2l1LmVkdS5hcjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMbzW / EpEv + qqZzfT1Buwjg9nnNNVrxkCfuR9fQiQw2tSouS5X37W5h7RmchRt54wsm046PDKtbSz1NpZT2GkmHN37yALW2lY7MyVUC7itv9vDAUsFr0EfKIdCKgxCKjrzkZ5ImbNvjxf7eA77PPGJnQ / UwXY7W + cvLkirp0K5uWpDk + nac5W0JXOCFR1BpPUJRbz2jFIEHyChRt7nsJZH6ejzNqK9lABEC76htNy1Ll / D3tUoPaqo8VlKW3N3MZE0DB9O7g65DmZIIlFqkaMH3ALd8adodJtOvqfDU / A6SxuwMfwDYPjoucykGDu1etRZ7dF2gd + W + 1Pn7yizPT1q8CAwEAAaNQME4wHQYDVR0OBBYEFPsn8tUHN8XXf23ig5Qro3beP8BuMB8GA1UdIwQYMBaAFPsn8tUHN8XXf23ig5Qro3beP8BuMAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQELBQADggEBAGu60odWFiK + DkQekozGnlpNBQz5lQ / bwmOWdktnQj6HYXu43e7sh9oZWArLYHEOyMUekKQAxOK51vbTHzzw66BZU91 / nqvaOBfkJyZKGfluHbD0 / hfOl / D5kONqI9kyTu4wkLQcYGyuIi75CJs15uA03FSuULQdY / Liv + czS / XYDyvtSLnu43VuAQWN321PQNhuGueIaLJANb2C5qq5ilTBUw6PxY9Z + vtMjAjTJGKEkE / tQs7CvzLPKXX3KTD9lIILmX5yUC3dLgjVKi1KGDqNApYGOMtjr5eoxPQrqDBmyx3flcy0dQTdLXud3UjWVW3N0PYgJtw5yBsS74QTGD4 = '
            );

            if (str_starts_with($config['idp']['x509cert'], 'file://')) {
                $config['idp']['x509cert'] = $this->extractCertFromFile($config['idp']['x509cert']);
            }
        }
        return $config;
    }

    /**
     * @throws Exception
     */
    protected function formatConfig(string $authenticatorName, array $config): array
    {
        $config = $this->formatIDPConfig($authenticatorName, $config);
        $urlParameters = $authenticatorName !== 'default' ? ['authenticator' => $authenticatorName] : [];

        if (empty($config['sp']['entityId'])) {
            $config['sp']['entityId'] = URL::route('saml2.metadata', $urlParameters);

            if ($config['sp']['forceHttps']) {
                $config['sp']['entityId'] = str_replace('http://', 'https://', $config['sp']['entityId']);
            }
        }
        if (empty($config['sp']['assertionConsumerService']['url'])) {
            $config['sp']['assertionConsumerService']['url'] = URL::route('saml2.acs', $urlParameters);

            if ($config['sp']['forceHttps']) {
                $config['sp']['assertionConsumerService']['url'] = str_replace('http://', 'https://', $config['sp']['assertionConsumerService']['url']);
            }
        }
        if (!empty($config['sp']['singleLogoutService']) && empty($config['sp']['singleLogoutService']['url'])) {
            $config['sp']['singleLogoutService']['url'] = URL::route('saml2.sls', $urlParameters);

            if ($config['sp']['forceHttps']) {
                $config['sp']['singleLogoutService']['url'] = str_replace('http://', 'https://', $config['sp']['singleLogoutService']['url']);
            }
        }
        if (str_starts_with($config['sp']['privateKey'], 'file://')) {
            $config['sp']['privateKey'] = $this->extractPkeyFromFile($config['sp']['privateKey']);
        }
        if (str_starts_with($config['sp']['x509cert'], 'file://')) {
            $config['sp']['x509cert'] = $this->extractCertFromFile($config['sp']['x509cert']);
        }

        Utils::setProxyVars($config['proxyVars']);

        if (!empty($config['sp']['assertionConsumerService']['url'])) {
            $params = ['authenticator' => $authenticatorName];
            if (!Str::contains($config['sp']['assertionConsumerService']['url'], http_build_query($params))) {
                // On passe le nom de l'authenticator a utilisé en query param
                $config['sp']['assertionConsumerService']['url'] .= '?'.http_build_query($params);
            }
        }

        return $config;
    }
}
