<?php

namespace Inside\Authentication\SAML\Http\Controllers;

use Closure;
use Exception;
use GuzzleHttp\Exception\RequestException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Inside\Authentication\Exceptions\AuthenticationException;
use Inside\Authentication\Exceptions\UserNotFoundException;
use Inside\Authentication\Facades\Authentication;
use Inside\Authentication\Models\User;
use Inside\Authentication\SAML\SAML\Authenticator;
use Inside\Authentication\SAML\SAML\AuthenticatorManager;
use Inside\Authentication\Services\AuthenticationService;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Inside\Permission\Facades\Permissions;
use Inside\Kernel\Authentication\Actions\SetMagicCookie;
use Inside\Kernel\Authentication\SingleSignOnProviderEnum;
use Laravel\Lumen\Http\Redirector;
use Laravel\Lumen\Http\ResponseFactory;
use OneLogin\Saml2\Error;

/**
 * Class Saml2Controller
 *
 * @package Inside\Authentication\SAML\Http\Controllers
 */
class Saml2Controller
{
    /**
     * Saml2Controller constructor.
     *
     * @param  AuthenticatorManager  $authenticatorManager
     */
    public function __construct(
        protected AuthenticatorManager $authenticatorManager
    ) {
    }

    /**
     *
     * Redirect to SAML login page
     * @throws Error
     */
    public function login(Request $request): void
    {
        /** @var string $authenticator */
        $authenticator = $request->query('authenticator', 'default');
        if (!$this->authenticatorManager->exists($authenticator)) {
            Log::error("Authenticator SAML \{$authenticator\} does not exist.");
            abort(404, "Authenticator SAML \{$authenticator\} does not exist.");
        }
        $authenticator = $this->authenticatorManager->get($authenticator);
        $authenticator->login($authenticator->config('loginRoute'));
    }


    /**
     * Output metadata depending on config to be parsed by SAML server
     *
     * @throws Error
     */
    public function metadata(Request $request): Response|ResponseFactory
    {
        /** @var string $authenticator */
        $authenticator = $request->query('authenticator', 'default');
        if (!$this->authenticatorManager->exists($authenticator)) {
            Log::error("Authenticator SAML \{$authenticator\} does not exist.");
            abort(404, "Authenticator SAML \{$authenticator\} does not exist.");
        }
        $authenticator = $this->authenticatorManager->get($authenticator);
        $metadata = $authenticator->getMetadata();

        return response($metadata, 200, ['Content-Type' => 'text/xml']);
    }

    /**
     * @throws AuthenticationException
     * @throws Exception
     */
    protected function handleAutoProvisioning(Authenticator $authenticator, array $attributes): User
    {
        // Création du user si l'auto_provisioning est activé
        Log::info('[SAML2 Authenticator] trying to create user');
        $bridge = new BridgeContent();
        $data = [];
        foreach ($authenticator->config('sync_attributes', '') as $insideField => $samlAttribute) {
            if ($samlAttribute instanceof Closure) {
                $data[$insideField] = $samlAttribute($authenticator, $bridge);
            } else {
                $value = $attributes[$samlAttribute];
                if (is_array($value) && count($value) === 1) {
                    $value = $value[0];
                }
                $data[$insideField] = $value;
            }
        }

        $validator = Validator::make(
            $data,
            [
                'name' => 'required|max:255',
                'mail' => 'required|email',
            ]
        );

        if ($validator->fails()) {
            Log::error('[SAML2 Authenticator] failed to create user (invalid name or email)');
            throw new AuthenticationException(
                Lang::get('auth.userCreationFailed'),
                [],
                config('saml2.info_url'),
                config('saml2.info_url_title')
            );
        }

        $data['status'] = 1;

        if (($uuid = $bridge->contentUpdate('users', $data)) === null) {
            Log::error('[SAML2 Authenticator] failed to create user (contentUpdate failed)');
            throw new AuthenticationException(
                Lang::get('auth.userCreationFailed'),
                [],
                config('saml2.info_url'),
                config('saml2.info_url_title')
            );
        }
        /** @var User $user */
        $user = User::withoutGlobalScopes()->find($uuid);

        return $user;
    }

    /**
     * SAML Login response treatment
     * @throws UserNotFoundException
     * @throws Exception
     */
    public function acs(Request $request): RedirectResponse|Redirector
    {
        /** @var string $authenticator */
        $authenticator = $request->query('authenticator', 'default');
        /** @var Authenticator $authenticator */
        $authenticator = $this->authenticatorManager->get($authenticator);
        $errors = $authenticator->acs();
        if (!empty($errors)) {
            Log::error('[SAML2 Authenticator] '.$authenticator->getLastErrorReason());

            throw new UserNotFoundException(
                Lang::get('auth.userDoesNotExist'),
                [],
                config('oauth2.info_url'),
                config('oauth2.info_url_title')
            );
        }
        $samlUser = $authenticator->getUser();

        if (config('saml2.binding.name_id_custom_attribute')) {
            $nameid = $samlUser->{config('saml2.binding.name_id_custom_attribute')};

            if (is_array($nameid)) {
                $nameid = $nameid[0];
            }
        } else {
            $nameid = $samlUser->{config('saml2.binding.name_id_method')}();
        }

        Log::debug('[SAML2 Authenticator] {'.$nameid.'}');

        $attributes = $authenticator->getAttributes();
        $authenticated = null;
        $discover = null;
        try {
            $model = $authenticator->config('binding.eloquent');
            $discover = $authenticator->config('binding.discover');
            /** @var User $test */
            $test = new $model();

            if ($test->getKeyName() === $discover) {
                $authenticated = Users::withoutGlobalScopes()->findOrFail($nameid);
            } else {
                $authenticated = Users::withoutGlobalScopes()->where($discover, $nameid)->firstOrFail();
            }

            /** @var User $authenticated */
            $authenticated = User::withoutGlobalScopes()->findOrFail($authenticated->uuid);

            Log::debug('[SAML2 Authenticator] {'.$authenticated->name.'} authenticated');
        } catch (Exception) {
            try {
                $samldiscover = $authenticator->config('binding.samldiscover');
                $attributeDiscover = $attributes[$samldiscover] ?? null;
                if (isset($samldiscover) &&
                    isset($attributeDiscover)) {
                    // get existing user if we get an array
                    if (is_array($attributeDiscover)) {
                        $authenticated = Users::withoutGlobalScopes()->whereIn($discover, $attributeDiscover)->first();

                        if (! $authenticated && $authenticator->config('auto_provisioning', false)) {
                            $authenticated = $this->handleAutoProvisioning($authenticator, $attributes);
                        }
                        if (!$authenticated) {
                            throw new ModelNotFoundException();
                        }
                    } elseif ($authenticator->config('auto_provisioning', false)) {
                        $authenticated = $this->handleAutoProvisioning($authenticator, $attributes);
                    } else {
                        $authenticated = Users::withoutGlobalScopes()->where($discover, $attributes[$samldiscover])->firstOrFail();
                    }
                    $authenticated = User::withoutGlobalScopes()->findOrFail($authenticated->uuid);
                } else {
                    throw new Exception();
                }
            } catch (ModelNotFoundException $e) {
                Log::error('[SAML2 Authenticator] '.$e->getMessage());
                $this->showError($e);
            }
        }

        if ($authenticated === null || $authenticated->status === false) {
            Log::error('[SAML2 Authenticator] user not found');
            throw new UserNotFoundException(
                Lang::get('auth.userDoesNotExist'),
                [],
                config('oauth2.info_url'),
                config('oauth2.info_url_title')
            );
        }

        // Redirect to front login with necessary informations
        $secure = Str::startsWith(env('APP_URL'), 'https://');

        (new SetMagicCookie())->execute($request, $authenticated, SingleSignOnProviderEnum::SAML);

        // Redirect to front login with necessary informations
        return redirect($authenticator->config('loginRoute'), 302, [], $secure);
    }

    /**
     * Logout from SAML
     * @throws Error
     * @throws Exception
     */
    public function logout(Request $request): void
    {
        $returnTo = $request->query('returnTo');
        $sessionIndex = $request->query('sessionIndex');
        $nameId = $request->query('nameId');

        $logoutService = new AuthenticationService();
        $logoutService->logout();

        /** @var string $authenticator */
        $authenticator = $request->query('authenticator', 'default');
        $authenticator = $this->authenticatorManager->get($authenticator);
        $authenticator->logout($returnTo, $nameId, $sessionIndex);
    }


    /**
     * SAML Logout response
     * @throws Exception
     */
    public function sls(Request $request): RedirectResponse|Redirector
    {
        /** @var string $authenticator */
        $authenticator = $request->query('authenticator', 'default');
        $authenticator = $this->authenticatorManager->get($authenticator);
        $error = $authenticator->sls($authenticator->config('retrieveParametersFromServer', false));
        if (!empty($error)) {
            throw new Exception("Could not log out");
        }

        return redirect($authenticator->config('logoutRoute', '/'));
    }


    /**
     * Show error
     * @throws Exception
     */
    protected function showError(Exception $e): void
    {
        if ($e instanceof RequestException) {
            abort(400, Lang::get('saml2.common_error', ['error' => $e->getMessage()]));
        }

        throw $e;
    }
}
