<?php

declare(strict_types=1);

namespace Inside\Authentication\Okta\Http\Controllers;

use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Authentication\Facades\Authentication;
use Inside\Authentication\Models\User;
use Inside\Kernel\Authentication\Actions\SetMagicCookie;
use Inside\Kernel\Authentication\SingleSignOnProviderEnum;
use Laravel\Socialite\Contracts\Provider;
use Laravel\Socialite\Facades\Socialite;
use SocialiteProviders\Manager\Config;
use SocialiteProviders\Okta\Provider as OktaProvider;
use Symfony\Component\HttpFoundation\RedirectResponse as SymfonyRedirectResponse;

final class LoginController extends Controller
{
    private function getProvider(): Provider
    {
        $configs = config('services.okta');

        /** @var OktaProvider $driver */
        $driver = Socialite::buildProvider(OktaProvider::class, $configs);

        $driver
            ->setConfig(new Config($configs['client_id'], $configs['client_secret'], $configs['redirect'], $configs))
            ->with(['state' => false])
            ->stateless();

        return $driver;
    }

    public function redirectToProvider(): SymfonyRedirectResponse
    {
        return $this->getProvider()->redirect();
    }

    public function handleProviderCallback(): RedirectResponse
    {
        $oktaUser = $this->getProvider()->user();

        $user = User::whereEmail($oktaUser->getEmail())->first();

        $secure = Str::startsWith(env('APP_URL'), 'https://');

        if (! $user instanceof User) {
            Log::error('[okta-authentication] User does not exist !', ['email' => $oktaUser->getEmail()]);
            return redirect('/login', 302, [], $secure);
        }

        (new SetMagicCookie())->execute(request(), $user, SingleSignOnProviderEnum::OKTA);

        return redirect('/external-login', 302, [], $secure);
    }
}
