<?php

namespace Inside\Authentication\OAuth2\Http\Controllers;

use Exception;
use GuzzleHttp\Exception\RequestException;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Authentication\Exceptions\UserNotActiveException;
use Inside\Authentication\Exceptions\UserNotFoundException;
use Inside\Authentication\Facades\Authentication;
use Inside\Authentication\Models\User;
use Inside\Kernel\Authentication\Actions\SetMagicCookie;
use Inside\Kernel\Authentication\SingleSignOnProviderEnum;
use Inside\Content\Models\Contents\Users;
use Laravel\Lumen\Http\Redirector;
use League\OAuth2\Client\Provider\Exception\IdentityProviderException;
use League\OAuth2\Client\Provider\GenericProvider as OAuth2Provider;
use League\OAuth2\Client\Token\AccessToken;

class OAuth2Controller
{
    /**
     * Login function for OAuth2
     */
    public function login(Request $request): RedirectResponse|Redirector
    {
        $code = $request->input('code');

        $provider = new OAuth2Provider(
            [
                'clientId'                => config('oauth2.client_id'),
                'clientSecret'            => config('oauth2.client_secret'),
                'redirectUri'             => config('oauth2.redirect_uri'),
                'urlAuthorize'            => config('oauth2.url_authorize'),
                'urlAccessToken'          => config('oauth2.url_access_token'),
                'urlResourceOwnerDetails' => config('oauth2.url_resource_owner_details'),
            ]
        );

        if ($code) {
            try {
                /** @var AccessToken $accessToken */
                $accessToken   = $provider->getAccessToken(
                    'authorization_code',
                    [
                        'code' => $code,
                    ]
                );

                $id_token = data_get($accessToken->getValues(), 'id_token', false);
                $token_field = config('oauth2.discover');
                $user_field = config('oauth2.user_discover');
                if (config('oauth2.url_resource_owner_details') === false || $id_token) {
                    [, $payload, ] = explode('.', $id_token);
                    $payload = strtr($payload, '-_', '+/');
                    $pad = strlen($payload) % 4;
                    if ($pad) {
                        $payload .= str_repeat('=', 4 - $pad);
                    }

                    $payload = json_decode(base64_decode($payload), true);
                    $upn = Arr::get($payload, $token_field);
                } else {
                    $resourceOwner = $provider->getResourceOwner($accessToken)->toArray();
                    $upn  = Arr::get($resourceOwner, $token_field);
                }

                /** @var ?string $uuid */
                $uuid = Users::where($user_field, $upn)->first()?->uuid;
                /** @var User|null $user */
                $user = User::find($uuid);

                Log::debug('[OAUTH2] {' . $upn . '}');

                // Check status
                if (!$user) {
                    throw new UserNotFoundException(
                        Lang::get('auth.userDoesNotExist'),
                        [],
                        config('oauth2.info_url'),
                        config('oauth2.info_url_title')
                    );
                }
                if (!$user->status) {
                    throw new UserNotActiveException(
                        Lang::get('auth.userIsNotActive'),
                        [],
                        config('oauth2.info_url'),
                        config('oauth2.info_url_title')
                    );
                }

                (new SetMagicCookie())->execute($request, $user, SingleSignOnProviderEnum::OAUTH2);

                // Redirect to front login with necessary informations
                $secure = Str::startsWith(env('APP_URL'), 'https://');

                return redirect(config('oauth2.login_route'), 302, [], $secure);
            } catch (Exception $e) {
                $this->showError($e);
            }
        }

        $options = config('oauth2.is_stable_api') ? ['scope' => config('oauth2.scope')] : [];
        $authorizationUrl = $provider->getAuthorizationUrl($options);


        return redirect($authorizationUrl);
    }

    /**
     * Logout from oauth2 endpoint
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse|\Laravel\Lumen\Http\Redirector|void
     */
    public function logout(Request $request)
    {
        return redirect(env('OAUTH2_URL_LOGOUT', env('APP_URL')));
    }

    /**
     * Show error
     * @throws Exception
     */
    protected function showError(Exception $e): void
    {
        if ($e instanceof RequestException) {
            abort(400, Lang::get('oauth2.common_error', ['error' => $e->getMessage()]));
        } elseif ($e instanceof IdentityProviderException) {
            abort(400, Lang::get('oauth2.oauth2_league', ['error' => $e->getMessage()]));
        }

        throw $e;
    }
}
