<?php

namespace Inside\Authentication\Ldap\Services;

use Adldap\Auth\BindException;
use Adldap\Auth\PasswordRequiredException;
use Adldap\Auth\UsernameRequiredException;
use Adldap\Laravel\Facades\Adldap;
use Adldap\Laravel\Facades\Resolver as LDAPResolver;
use Adldap\Models\Model as AdldapModel;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Lang;
use Inside\Authentication\Exceptions\PasswordNotValidException;
use Inside\Authentication\Exceptions\UserNotFoundException;
use Inside\Authentication\Ldap\Contracts\AuthenticationLdap;
use Inside\Authentication\Models\User;
use Inside\Support\Str;

/**
 *
 * @category Class
 * @package  Inside\Authentication\Ldap\Services\InsideAuthenticationLdapServiceProvider
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class AuthenticationLdapService implements AuthenticationLdap
{
    /**
     * Connect with $credentials
     * @throws PasswordNotValidException
     * @throws UserNotFoundException
     * @throws BindException
     * @throws PasswordRequiredException
     * @throws UsernameRequiredException
     */
    public function connect(array $credentials): User
    {
        $user = $this->getUserFromCredentials($credentials);
        if ($user === null) {
            throw new UserNotFoundException(Lang::get('auth.userDoesNotExist'));
        }
        $attribute = LDAPResolver::getLdapDiscoveryAttribute();
        $search = $this->getDiscoveryAttributeValueFromCredentials($credentials);
        if ($search === null) {
            throw new UserNotFoundException(Lang::get('auth.userDoesNotExist'));
        }

        Adldap::setDefaultProvider(strtolower($user->provider_name ?? 'default'));
        // Get ldap user by its discover attribute
        $ldapUser = Adldap::search()->where($attribute, $search)->first();
        if (!$ldapUser instanceof AdldapModel) {
            throw new UserNotFoundException(Lang::get('auth.userDoesNotExist'));
        }

        // Now let's connect with our auth attribute !
        $attribute = LDAPResolver::getLdapAuthAttribute();
        if ($attribute == 'dn') {
            $attribute = $ldapUser->getSchema()->distinguishedName();
        }

        $username = $ldapUser->getFirstAttribute($attribute);
        $usernames = [];

        if (! Str::contains($username, ',')) {
            $usernames = [$username];
        } else {
            if ($attribute == 'cn') {
                // Special fix when cn contains ','
                $usernames[] = str_replace(',', '\,', $username);
            } elseif ($attribute === $ldapUser->getSchema()->distinguishedName()) {
                // Rebuild dn from cn
                $cn = str_replace(',', '\,', $ldapUser->getFirstAttribute('cn'));

                $usernames = collect(config('ldap.connections'))
                    ->pluck('settings.base_dn')
                    ->transform(
                        fn ($base_dn) => is_null($base_dn) ? "CN={$cn}" : "CN={$cn},{$base_dn}"
                    )->toArray();
            }
        }

        $password = $this->getPasswordFromCredentials($credentials);

        foreach ($usernames as $username) {
            if (Adldap::auth()->attempt($username, $password, config('bind.bind_as_user'))) {
                return $user;
            }
        }

        throw new PasswordNotValidException(Lang::get('auth.incorrectPassword'));
    }

    /**
     * get password from $credentials
     *
     * @param array $credentials
     * @return string
     */
    protected function getPasswordFromCredentials(array $credentials): string
    {
        return Arr::get($credentials, 'password');
    }

    /**
     * get Discovery attribute to match ldap user from $credentials
     *
     * @param array $credentials
     * @return string|null
     */
    protected function getDiscoveryAttributeValueFromCredentials(array $credentials): ?string
    {
        $fieldName = LDAPResolver::getDatabaseUsernameColumn();
        return Arr::get($credentials, $fieldName);
    }

    /**
     * Get user from $credentials
     *
     * @param array $credentials
     * @return User|null
     */
    protected function getUserFromCredentials(array $credentials): ?User
    {
        $fieldName = LDAPResolver::getDatabaseUsernameColumn();
        $fieldValue = Arr::get($credentials, $fieldName);
        if ($fieldValue === null) {
            return null;
        }

        /** @var ?User $user */
        $user = User::query()->where($fieldName, $fieldValue)->first();

        return $user;
    }
}
