<?php

declare(strict_types=1);

namespace Inside\Authentication\Ldap\Http\Controllers;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inside\Authentication\Contracts\Authentication;
use Inside\Authentication\Ldap\Contracts\AuthenticationLdap;
use Inside\Authentication\Models\User;
use Inside\Authentication\Facades\TwoFactorAuthentication;

/**
 * @category Class
 * @package  Inside\Authentication\Ldap\Controllers\AuthenticationWindowsController
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class Login
{
    public function __invoke(
        Request $request,
        AuthenticationLdap $authenticationLdap,
        Authentication $authentication
    ): JsonResponse {
        $credentials = $request->only('name', 'email', 'password');
        $remember = $request->get('remember', false);

        try {
            $user = $authenticationLdap->connect($credentials);
            $clientIp = $request->ip();
            $authorizedIps = explode(',', env('2FA_AUTHORIZED_IPS', ''));
            $isClientWhiteListed = in_array(
                $request->ip(),
                $authorizedIps
            ) || TwoFactorAuthentication::checkClientIpInGivenRanges($clientIp);

            if (env('2FA_ENABLED') && !$isClientWhiteListed) {
                try {
                    TwoFactorAuthentication::sendOtpToken($user);
                } catch (Exception $ex) {
                    return new JsonResponse([
                        'message' => $ex->getMessage(),
                    ], 500);
                }

                return response()->json([
                    'data' => [
                        'uuid' => $user->uuid,
                        '2fa_enabled' => true,
                    ],
                ]);
            }

            $token = $authentication->logAs($user, 'ldap', $remember);
        } catch (Exception) {
            // Fall back to inside connection
            $token = $authentication->login($credentials, $remember);
        }
        $user = $token->accessToken->user;

        $transformed = $user->toArray();
        $transformed['api_token'] = $token->plainTextToken;

        // Set header so we can use auth api guard
        $request->headers->set('api_token', $token->plainTextToken);
        return response()->json([
            'data' => $transformed,
        ]);
    }
}
