<?php

namespace Inside\Authentication\Google\Http\Controllers;

use Google\Service\PeopleService;
use Illuminate\Http\RedirectResponse;
use Inside\Authentication\Facades\Authentication;
use Inside\Authentication\Models\User;
use Inside\Kernel\Authentication\Actions\SetMagicCookie;
use Inside\Kernel\Authentication\SingleSignOnProviderEnum;
use Laravel\Lumen\Http\Redirector;
use Laravel\Socialite\Facades\Socialite;
use Laravel\Socialite\Two\GoogleProvider;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use Symfony\Component\HttpFoundation\RedirectResponse as BaseRedirectResponse;

class LoginController
{
    /**
     * Redirect the user to google login form.
     *
     * @param Request $request
     * @return RedirectResponse|BaseRedirectResponse
     */
    public function redirectToProvider(Request $request): RedirectResponse|BaseRedirectResponse
    {
        $domain = $request->input('domain');
        $allConfigs = Config::get('google.google_domains');

        if (!empty($allConfigs) && !empty($allConfigs[$domain])) {
            $configs = array_merge(Config::get('services.google'), $allConfigs[$domain]);
            $driver = Socialite::buildProvider(GoogleProvider::class, $configs);
        } else {
            $driver = Socialite::driver('google');
            $domain = env('GOOGLE_DOMAIN');
        }

        $driver
            ->stateless()
            ->scopes(
                [
                    'openid',
                    'profile',
                    'email',
                    PeopleService::CONTACTS_READONLY,
                ]
            );

        if ($domain) {
            $driver->with(['hd' => $domain]);
        }

        return $driver->redirect();
    }

    /**
     * Really log into inside
     *
     * @param Request $request
     * @return RedirectResponse|Redirector
     */
    public function handleProviderCallback(Request $request): RedirectResponse|Redirector
    {
        $domain = $request->input('hd');
        $allConfigs = Config::get('google.google_domains');

        if (!empty($allConfigs) && !empty($allConfigs[$domain])) {
            $configs = array_merge(Config::get('services.google'), $allConfigs[$domain]);
            $driver = Socialite::buildProvider(GoogleProvider::class, $configs);
        } else {
            $driver = Socialite::driver('google');
        }
        $user = $driver->stateless()->user();
        $user = User::where('email', '=', $user->getEmail())->first();

        $secure = Str::startsWith(env('APP_URL'), 'https://');

        if (! $user instanceof User) {
            Log::error('[google-authentication] User does not exist !');

            return redirect('/login', 302, [], $secure);
        }

        (new SetMagicCookie())->execute($request, $user, SingleSignOnProviderEnum::GOOGLE);

        return redirect('/external-login', 302, [], $secure);
    }
}
